#ifndef mst_rdb_h_INCLUDED
#define mst_rdb_h_INCLUDED

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of mst_rdb
 *
 * mst_rdb is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * mst_rdb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <stdio.h>


#ifdef __cplusplus
extern "C" {
#endif
  
typedef struct
{
  unsigned long index;
  unsigned long column;
} DataColumnMapping;

typedef struct
{
  unsigned long n;
  DataColumnMapping *map;
} DataColumnMap;

typedef enum 
{
   RDB_String = 0,
   RDB_Num
} RDB_Type;

typedef struct
{
  unsigned long n;	       /* number of columns */
  char *names;		       /* a buffer containing the column names */
  char **col_name;	       /* an array of pointers to the column names */
  unsigned long line;	       /* line number in file */
  int can_rewind;		/* can the file be rewound? */
  long	start;		       /* position in file marking the beginning of the
			          first row of data */
  void *buf;		       /* the fget_rec buffer */
} rdbHeader;

typedef struct
{
  size_t offset; 	       /* where to put the data in the target struct */
  unsigned long column;        /* where to find the data in the file */
  RDB_Type type;	       /* 0=string, 1=double */
} DataColumnMapping_st;

typedef struct
{
  unsigned long n;             /* number of columns to be read */
  DataColumnMapping_st *map;   /* bunch of mappings; one per column */
} DataColumnMap_st;

typedef struct
{
  void *data;                  /* where to put the data */
  unsigned long column;        /* where to find the data in the file */
  int type;                    /* 0=string, 1=double */
} DataColumnMapping_ar;

typedef struct
{
  unsigned long n;
  DataColumnMapping_ar *map;
} DataColumnMap_ar;

typedef struct
{
  const char *name;            /* name of rdb column */
  RDB_Type type;               /* data type; 0=string, 1=double */
  size_t offset;               /* offset into user's structure */
} RDBFieldStInfo;

#define RDBentry( name, type, target ) { #name, type, offsetof( target , name ) }

rdbHeader *
rdb_rd_hdr(
  FILE *fin                     /* the stream to parse */
);

void
rdb_free_hdr(
  rdbHeader *hdr                /* the header to free */
);

DataColumnMap_st * 
rdb_map_cols_arst(
  rdbHeader const *hdr,         /* the rdb header */
  unsigned long ncols,          /* the number of columns to read */
  char *col[],                  /* an array of pointers to strings holding the
                                   names of the columns to map */
  RDB_Type const type[],        /* an array of type specifiers; allowed
                                   values are RDB_String, RDB_Num 
                                   (i.e. double) */
  size_t const data_offset[]    /* offsets into struct for data values */
);

DataColumnMap_st * 
rdb_map_cols_stst(
  rdbHeader const *hdr,         /* the rdb header */
  unsigned long ncols,          /* the number of columns to read */
  RDBFieldStInfo const fields[] /* an array of RDBFieldStInfo structs
                                   containing names of the columns to 
                                   map, data types (string or numeric), 
                                   and offsets into the user-supplied 
                                   target struct for each column to be read */
);

int
rdb_is_column( 
  rdbHeader const *hdr,        /* header to be searched */
  char const      *name        /* header name sought */
);

int
rdb_col_read_st(
  FILE *fin,                    /* the stream to parse */
  rdbHeader *hdr,               /* the rdb file description */
  DataColumnMap_st const *map,  /* the columns to read */
  void * data                   /* where to stow the data. must be pointer
                                   to an appropriate struct */
);

long
rdb_count(
  FILE *fin,                    /* the stream to parse */
  rdbHeader *hdr                /* the rdb file description */
);

void
rdb_free_map(
  DataColumnMap_st *map /* the column map to free */
);

int
rdb_rewind(
  FILE *fin,                    /* the stream to parse */
  rdbHeader *hdr                /* the rdb file description */
);

#ifdef __cplusplus
}
#endif

#endif /* mst_rdb_h_INCLUDED */
