#ifndef RDBColumnTmplt_h
#define RDBColumnTmplt_h

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of RDB
 *
 * RDB is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * RDB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <sstream>
#include <string>
#include <utility>

#include <rdbxx/RDBColumn.h>

/// Parameterizes RDBColumn interface for many data types.
template <class Tmplt0, class Tmplt1, class Tmplt2>
class RDBColumnTmplt : public RDBColumn {
 public:
  /// @name Constructing, destructing, and initializing RDB columns.
  //@{
  /// Assigns name and definition.
  RDBColumnTmplt( const std::string& name="", const std::string& def="" );
  /// Copies RDBColumnTmplt object.
  RDBColumnTmplt( const RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>& rdbcolumntmplt );
  /// Deletes resources allocated by RDBColumnTmplt object.
  ~RDBColumnTmplt( void );

  /// Copies RDBColumnTmplt object.
  RDBColumn& operator=( const RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>& rdbcolumntmplt );
  /// Assigns data to RDBColumn object's _data member, converting as necessary.
  virtual RDBColumn& operator=( const Tmplt0& data );
  /// Assigns data to RDBColumn object's _data member, converting as necessary.
  virtual RDBColumn& operator=( const Tmplt1& data );
  /// Assigns data to RDBColumn object's _data member, converting as necessary.
  virtual RDBColumn& operator=( const Tmplt2& data );
  //@}

  /// @name Auto-indexing control methods.
  //@{
  /// Increments index to the RDBColumn's data elements.
  virtual void advanceIdx( void );
  /// Rewinds index to the RDBColumn's data elements.
  virtual void rewind( void );
  //@}

  /// @name Group information ("break" column) methods.
  //@{
  virtual void setGroup( bool group );
  /// Sets the group value to the current data value.
  virtual void setGroupValue( void );
  /// Returns the group status of this column object.
  virtual int newGroup( void );
  //@}
  
  /// @name Data member initializers.
  //@{
  /// Sets the data value, converting as necessary.
  virtual bool setData( const Tmplt0& data );
  /// Sets the data value, converting as necessary.
  virtual bool setData( const Tmplt1& data );
  /// Sets the data value, converting as necessary.
  virtual bool setData( const Tmplt2& data );
  //@}

  /// @name Methods to map RDBColumn's data to user-supplied memory.
  //@{
  /// Maps data to user-supplied memory.
  virtual void mapData( Tmplt0 data[], const size_t nelems );
  //@}

  
  /// @name Data member accessors.
  //@{
  /// Returns a pointer to the current data element.
  void* getData( void );
  /// Returns the value of the current data element, converting if necessary.
  virtual bool getData( Tmplt0& data );
  /// Returns the value of the current data element, converting if necessary.
  virtual bool getData( Tmplt1& data );
  /// Returns the value of the current data element, converting if necessary.
  virtual bool getData( Tmplt2& data );
  /// Returns the value of the current data element, converting if necessary.
  virtual double getDataDouble( void );
  /// Returns the value of the current data element, converting if necessary.
  virtual long getDataLong( void );
  /// Returns the value of the current data element, converting if necessary.
  virtual std::string getDataString( void );
  //@}

 protected:
  /// Called by the stream insertion operator.
  virtual std::istream& read( std::istream& is );
  /// Called by the stream extraction operator.
  virtual std::ostream& write( std::ostream& os ) const;
  /// Deletes resources allocated by RDBColumnTmplt object.
  void cleanup( void );

  /// Pointer to the data managed by object.
  Tmplt0* _data;
  /// Index into the data.
  size_t _idx;
  /// Number of elements of data.
  size_t _nelems;
  /// Indicates that RDBColumnTmplt is responsible for deallocating the data.
  bool _mine;

  /// Current group value.
  Tmplt0 _groupvalue;
  
};

#ifndef RDBColumnTmplt_cc
#include <rdbxx/RDBColumnTmplt.cc>
#endif


/// Provide user with a double column capable of converting to long or string.
typedef RDBColumnTmplt<double,long,std::string> RDBDoubleColumn;
/// Provide user with a long column capable of converting to double or string.
typedef RDBColumnTmplt<long,std::string,double> RDBLongColumn;
/// Provide user with a string column capable of converting to double or long.
typedef RDBColumnTmplt<std::string,double,long> RDBStringColumn;

#endif
