#ifndef rl_DielectricPODArray_h_INCLUDED
#define rl_DielectricPODArray_h_INCLUDED

// File:   rl_DielectricPODArray.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Note       : (1-alpha) is real part of complex dielectric constant
 *              (-gamma)  is imag part of complex dielectric constant
 */

#include <cstddef>                   // size_t
#include <rl_raylib/rl_Traits.h>     // rl_Traits::rl_DielectricPOD
#include <rl_raylib/rl_Exception.h>  // rl_Exception

//########################################################################
// rl_DielectricPODArray
//########################################################################
//
/**
 * \class rl_DielectricPODArray
 *
 * A class encapsulating an array of rl_DielectricPODs storing the
 * dielectric contants as a function of energy:
 * - energy (keV)
 * - alpha (real part of dielectric decrement)
 * - gamma (imaginary part of the dielectric decrement)
 *
 * The complex dielectric constant has real part (1-alpha) and 
 * imaginary part (-gamma).
 */

class rl_DielectricPODArray
{
protected:
 
  /// number of dielectric decrements read in
  size_t                       nelts_;
  /// pointer to the data
  rl_Traits::rl_DielectricPOD* data_;

public:

  /**
   * Destructor
   */
 ~rl_DielectricPODArray();

  /**
   * Default constructor.
   *
   *     An empty uninitialized rl_DielectricPODArray is created 
   *     and the init method must be called to initialize the object.
   */
  rl_DielectricPODArray();

  /**
   * Constructor.
   *
   * @param nelts number of elements in the array
   * @param energy array of energies
   * @param alpha  array of dielectric decrement real part (alpha)
   * @param gamma  array of dielectric decrement imag part (gamma)
   */
  rl_DielectricPODArray( size_t        nelts, 
                         double const* energy,
                         double const* alpha,
                         double const* gamma );

  /**
   * Constructor.
   *
   * @param nelts number of elements in the array
   * @param diel array of dielectric decrement PODs
   */
  rl_DielectricPODArray( size_t                       nelts, 
                         rl_Traits::rl_DielectricPOD* diel
                       );

  /**
   * Initializer.
   *
   * @param nelts number of elements in the array
   * @param energy array of energies
   * @param alpha  array of dielectric decrement real part (alpha)
   * @param gamma  array of dielectric decrement imag part (gamma)
   *
   * initialize this rl_DielectricPODArray from the energy, 
   * alpha, and gamma arrays.  The rl_DielectricPODArray is
   * sorted on the energy field.
   */
  void
  init( size_t        nelts, 
        double const* energy,
        double const* alpha,
        double const* gamma );

  /**
   * Initializer.
   *
   * @param nelts number of elements in the array
   * @param diel array of dielectric decrement PODs
   *
   * initialize this rl_DielectricPODArray from the
   * input array of rl_DielectricPODs.  The rl_DielectricPODArray 
   * is sorted on the energy field.
   */
  void
  init( size_t                       nelts, 
        rl_Traits::rl_DielectricPOD* diel );

  /**
   * Accessor.
   *
   * @return number of elements in the rl_DielectricPOD array.
   */
  size_t num_elts() const;

  /**
   * Accessor.
   *
   * @return  pointer-to-const to rl_DielectricPOD array.
   */
  rl_Traits::rl_DielectricPOD const* const_data_ptr() const;

  /**
   * Accessor.
   *
   * @return  pointer-to-const to rl_DielectricPOD array.
   */

  /**
   * Print reflectivity information to output FILE* stream.
   *
   * @param of output FILE* stream.
   * @param pre optional prefix (char*) string.
   * @param pst optional postfix (char*) string.
   */
  void cprint_on( std::FILE* of, char const pre[] = "",
                                 char const pst[] = "" ) const;
};

inline size_t rl_DielectricPODArray::
num_elts() const
{ return nelts_; }

inline rl_Traits::rl_DielectricPOD const* rl_DielectricPODArray::
const_data_ptr() const
{ return data_; }

// rl_DielectricPODArray_h_INCLUDED
#endif
