#ifndef rl_TransmissionCoefPOD_h_INCLUDED
#define rl_TransmissionCoefPOD_h_INCLUDED

// File:   rl_transmissioncoefpod.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Description: declare rl_TransmissionCoefPOD
 *
 * rl_TransmissionCoeff contains the in-plane and out-of-plane reflection
 * coefficients;
 *   PolPlus:    '+', or parallel      (in-plane),     reflection coefficient
 *   PolMinus:   '-', or perpendicular (out-of-plane), reflection coefficient
 *
 * Here, in-plane means the E-vector is in the vector in the plane 
 * of incidence (defined by the ray direction and the surface normal),
 * while out-of-plane means the E-vector is perpendicular to the 
 * plane of incidence.
 *
 * NOTE:  a POD class must satisfy (Jack W. Reeves, C++Report, February 1998,
 *         p. 46-54):
 *  - no user-declared constructors, copy-assign operator, or destructor
 *  - no private or protected nonstatic data members
 *  - no base classes
 *  - no virtual functions
 *  - no nonstatic data members of type pointer to member,
 *       non-POD-struct/class (or array of such types) or reference
 *
 * History
 *--------
 * 1.0.1 2004-Dec-07  tjg  simplify/remove include guards
 * 0.2.0 1998-Jun-24  tjg  C++ version
 * 0.1.0 1998-Jun-23  tjg  revised notation; renamed file
 * 0.0.0 1995-May-18  tjg  original version
 */

#include <rl_raylib/rl_Traits.h>
#include <cstring>                  // strlen
#include <iostream>                 // <<

//########################################################################
// rl_TransmissionCoefPOD
//########################################################################
//
/** 
 * \class rl_TransmissionCoefPOD
 *
 * A Plain Ol' Data class representing complex reflection coefficients.
 */

class rl_TransmissionCoefPOD
{
private:

  /// '+', or in-plane, reflection coefficient
  rl_Traits::complex para_;  
  /// '-', or out-of-plane, reflection coefficient
  rl_Traits::complex perp_;

public:

  /**
   * Initialize perpendicular (s) and parallel (p) transmission coefficients
   * to zero.
   */
  void init();

    
  /**
   * Iinitialize perpendicular (s) and parallel (p) transmission coefficients.
   *
   * @param para parallel transmission coefficient
   * @param perp perpendicular transmission coefficient
   */
  void init( rl_Traits::complex& para,
             rl_Traits::complex& perp );

  /**
   * Transmission factor.
   *
   * @param polarization_factor polarization factor; it must be
   *        a value between -1 and 1.  The polarization factor is 
   *        related to parallel (p) and perpendicular (s) polarization by:
   *  \f[
   *      \mathtt{polarization\_factor} = {(I_\perp - I_\parallel) 
   *                                     \over 
   *                                     (I_\perp + I_\parallel)} 
   *  \f]
   * or
   *  \f[
   *      \mathtt{polarization\_factor} = {(I_\mathit{s} - I_\mathit{p}) 
   *                                     \over 
   *                                     (I_\mathit{s} + I_\mathit{p})} 
   *  \f]
   * where \f$I_\perp\f$ and \f$I_\parallel\f$ are the perpendicular
   * and parallel E-field \em intensities, respectively.  Thus,
   *  - -1: pure parallel (p) polarization.
   *  - 0: completely unpolarized.
   *  - +1: pure perpendicular (s) polarization.
   *
   * @return transmission factor
   */
  double transmission( double polarization_factor = 0.0 ) const;

  /**
   * @return parallel (p) transmission coefficient.
   */
  rl_Traits::complex para() const;

  /**
   * @return parallel (p) transmission coefficient (read/write access).
   */
  rl_Traits::complex& para();

  /**
   * @return perpendicular (s) transmission coefficient.
   */
  rl_Traits::complex perp() const;

  /**
   * @return perpendicular (s) transmission coefficient (read/write access).
   */
  rl_Traits::complex& perp();

  /**
   * Print reflectivity information to output stream.
   *
   * @param os output stream.
   * @param pre optional prefix (char*) string.
   * @param pst optional postfix (char*) string.
   */
  std::ostream& print_on( std::ostream& os, char const pre[] = "",
                                            char const pst[] = "" ) const;

  /**
   * Print reflectivity information to output FILE* stream.
   *
   * @param of output FILE* stream.
   * @param pre optional prefix (char*) string.
   * @param pst optional postfix (char*) string.
   */
  void cprint_on( std::FILE* of, char const pre[] = "",
                                 char const pst[] = "" ) const;
};

inline void rl_TransmissionCoefPOD::
init()
{
  para_ = rl_Traits::complex();
  perp_ = rl_Traits::complex();
}

inline void rl_TransmissionCoefPOD::
init( rl_Traits::complex& para,
      rl_Traits::complex& perp )
{
  para_ = para;
  perp_ = perp;
}

inline rl_Traits::complex rl_TransmissionCoefPOD::
para() const
{ return para_; }

inline rl_Traits::complex rl_TransmissionCoefPOD::
perp() const
{ return perp_; }

inline rl_Traits::complex& rl_TransmissionCoefPOD::
para()
{ return para_; }

inline rl_Traits::complex& rl_TransmissionCoefPOD::
perp()
{ return perp_; }

inline double rl_TransmissionCoefPOD::
transmission( double polarization_factor ) const
{
  return ( norm( perp_ ) * (1.0 + polarization_factor ) 
         + norm( para_ ) * (1.0 - polarization_factor ) ) / 2.0;
}

inline std::ostream& rl_TransmissionCoefPOD::
print_on( std::ostream& os, char const pre[], char const pst[] ) const
{
  if ( std::strlen(pre) ) { os << pre; }
  os << "[" << para_ << "][" << perp_ << "]";
  if ( std::strlen(pst) ) { os << pst; }
  return os;
}

inline void rl_TransmissionCoefPOD::
cprint_on( std::FILE* of, char const pre[], char const pst[] ) const
{
  if ( std::strlen(pre) ) { std::fprintf(of, "%s", pre); }
  std::fprintf(of, "[%.15e, %.15e] [%.15e, %.15e]\n",
     para_.real(), para_.imag(), perp_.real(), perp_.imag());
  if ( std::strlen(pst) ) { std::fprintf(of, "%s", pst); }
}

inline std::ostream&
operator<<( std::ostream& os, rl_TransmissionCoefPOD const& rfl )
{ rfl.print_on( os ); return os; }

// rl_TransmissionCoefPOD_h_INCLUDED
#endif
