#ifndef SUPLIB_RANGE_H
#define SUPLIB_RANGE_H

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of suplib
 *
 * suplib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * suplib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <stdio.h>
#include <stddef.h>
#include <limits.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#define RANGEL_MAX	LONG_MAX
#define RANGEL_MIN	LONG_MIN

#define RANGEF_MAX	DBL_MAX
#define RANGEF_MIN	(-DBL_MIN)

typedef enum
{
  RangeErr_OK = 0,		/* no error */
  RangeErr_NOMEM,		/* out of memory */
  RangeErr_INCOMPLETE,		/* incomplete range */
  RangeErr_ERANGE,		/* number out of bounds */
  RangeErr_ILLNUM,		/* not a number */
  RangeErr_NEGNUM,		/* negative number */
  RangeErr_OFLOWSTART,		/* overflow of start value */
  RangeErr_NONPOSCOUNT,		/* non-positive count */
  RangeErr_OFLOWEND,		/* overflow of end value */
  RangeErr_INTERNAL,		/* internal error */
  RangeErr_ORDER,		/* start greater than end */
  RangeErr_EMPTY,		/* float range is empty set */
  RangeErr_MAXERR
} RangeErr;

typedef enum
{
  Range_SORT       = 0x0001,
  Range_MERGE      = 0x0002,
  Range_INCOMPLETE = 0x0004,
  Range_UNSIGNED   = 0x0008
} RangeOpts;

/* float range operands */
typedef enum
{
  RangeOp_INCLUSIVE = 0x00,	/* both start and end inclusive */
  RangeOp_START     = 0x01,	/* start exclusive */
  RangeOp_END       = 0x02	/* end exclusive */
} RangeOp;

typedef struct
{
  long start;
  long end;
} RangeL;

typedef struct
{
  RangeL *range;
  RangeL *pos;
  size_t n;
  long	val;
} RangeLList;

typedef struct
{
  float start;
  float end;
  RangeOp op;
} RangeF;

typedef struct
{
  RangeF *range;
  int	bsearch;
  size_t n;
} RangeFList;


RangeErr	rangel_parse(RangeLList **rll, char *range_spec,
			     RangeOpts opts, long minval, long maxval,
			     long *where);

RangeLList *	rangel_new(size_t nr);
RangeLList *	rangel_del(RangeLList *rl);
void		rangel_reset(RangeLList *rl);
int		rangel_next(RangeLList *rl, long *elem);
int		rangel_minmax(RangeLList *rl, long *min, long *max );

unsigned long	rangel_count(RangeLList *rl);
int		rangel_check(RangeLList *rl, long minval, long maxval);
void		rangel_rep_minval(RangeLList *rl, long minval, long repval);
void		rangel_rep_maxval(RangeLList *rl, long maxval, long repval);
void		rangel_rep_val(RangeLList *rl, long minval, long maxval,
			       long minrepval, long maxrepval);
void		rangel_dump(FILE *fout, RangeLList *rl);

void		range_perror(FILE *fout, RangeErr error, const char *spec,
			      long where);

RangeFList *	rangef_new(size_t nr);
RangeFList *	rangef_del(RangeFList *rl);
RangeErr	rangef_parse( RangeFList **rfl, char *range_spec,
			      RangeOpts opts, double minval, double maxval,
			      long *where);
    int		rangef_in(RangeFList *rl, double t);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* ! SUPLIB_RANGE_H */
