# --8<--8<--8<--8<--
#
# Copyright (C) 2006 Smithsonian Astrophysical Observatory
#
# This file is part of SAOsac::gi
#
# SAOsac::gi is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# SAOsac::gi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
#       Free Software Foundation, Inc.
#       51 Franklin Street, Fifth Floor
#       Boston, MA  02110-1301, USA
#
# -->8-->8-->8-->8--

package SAOsac::gi;

use strict;
use warnings;

use Carp;
use IO::File;

our $VERSION = '0.08';


my %field = (

    'eps_i' => {
        'comment' => 'imaginary part of dielectric constant',
        'id'      => 15
    },
    'z0' => {
        'comment' => 'z coordinate of surface center',
        'id'      => 3
    },
    'ri' => {
        'comment' => 'inner radius of aperture annulus',
        'id'      => 9
    },
    'y0' => {
        'comment' => 'y coordinate of surface center',
        'id'      => 2
    },
    'rho0' => {
        'comment' => 'surface center radius',
        'id'      => 6
    },
    'x0' => {
        'comment' => 'x coordinate of surface center',
        'id'      => 1
    },
    'dfm_type' => {
        'comment' => 'deformation type: ',
        'id'      => 30
    },
    'rings' => {
        'comment' => 'number of rings',
        'id'      => 11
    },
    'ro' => {
        'comment' => 'outer radius of aperture annulus',
        'id'      => 10
    },
    'bundle_el' => {
        'comment' => 'incoming bundle elevation angle',
        'id'      => 8
    },
    'src_dist' => {
        'comment' => 'source distance',
        'id'      => 13
    },
    'k' => {
        'comment' => 'surface factor',
        'id'      => 5
    },
    'zoff' => {
        'comment' => 'z offset of focal plane',
        'id'      => 3,
        format    => '%.20g',
    },
    'optic_type' => {
        'comment' => 'optic type: ',
        'id'      => 29
    },
    'el_mis' => {
        'comment' => 'surface misalignment: elevation',
        'id'      => 8
    },
    'bundle_az' => {
        'comment' => 'incoming bundle azimuth angle',
        'id'      => 7
    },
    'spokes' => {
        'comment' => 'number of spokes',
        'id'      => 12
    },
    'zoff_origin' => {
        'comment' => 'zoff measured wrt: ',
        'id'      => 23
    },
    'ns' => {
        'comment' => 'number of surfaces',
        'id'      => 4
    },
    'az_mis' => {
        'comment' => 'surface misalignment: azimuth',
        'id'      => 7
    },
    'l' => {
        'comment' => 'surface length',
        'id'      => 10
    },
    'p' => {
        'comment' => 'surface conic constant',
        'id'      => 4
    },
    'eps_r' => {
        'comment' => 'real part of dielectric constant',
        'id'      => 14
    },
    'dfm' => {
        'comment' => 'surface deformation: ',
        'id'      => 22
    } );


# Preloaded methods go here.

sub create {
    my ( $gi, $comment, $system, @surfaces ) = @_;


    # create .gi file
    my $fh = IO::File->new( $gi, 'w' )
      or croak( "couldn't create gi file `$gi'\n" );

    print $fh $comment, "\n";

    while ( my ( $key, $value ) = each %$system ) {

        if ( $key eq 'zoff_origin' ) {
            gi( $fh, 0, $key, $value, $value ? 'STD' : 'GFOC' );
        }

        else {

            gi( $fh, 0, $key, $value );

        }

    }

    my $srfno = 0;

    for my $surface ( @surfaces ) {

        $srfno = $surface->{srfno} // $srfno + 1;

        shell( $fh, $srfno, $surface );

    }

    print $fh "-1,0,0\n";
    $fh->close;
}

sub shell {
    my ( $fh, $shell, $info ) = @_;

    gi( $fh, $shell, 'x0',     $info->{'x0'} );
    gi( $fh, $shell, 'y0',     $info->{'y0'} );
    gi( $fh, $shell, 'z0',     $info->{'z0'} );
    gi( $fh, $shell, 'p',      $info->{'p'} );
    gi( $fh, $shell, 'k',      $info->{'k'} );
    gi( $fh, $shell, 'rho0',   $info->{'rho0'} );
    gi( $fh, $shell, 'az_mis', $info->{'az_mis'} );
    gi( $fh, $shell, 'el_mis', $info->{'el_mis'} );
    gi( $fh, $shell, 'l',      $info->{'l'} );

    gi( $fh, $shell, 'eps_r', $info->{'eps_r'} ) if exists $info->{'eps_r'};

    gi( $fh, $shell, 'eps_i', $info->{'eps_i'} ) if exists $info->{'eps_i'};

    gi( $fh, $shell, 'dfm', $info->{'dfm'}, $info->{'dfm'} ? 'yes' : 'no' );

    gi( $fh, $shell, 'dfm_type', $info->{'dfm_type'},
        $info->{'dfm_type'} ? 'spline' : 'cogen' );


    gi( $fh, $shell, 'optic_type', $info->{'optic_type'},
        $info->{'optic_type'} ? 'xray' : 'conventional' );
}

sub gi {
    my ( $fh, $rec_type, $param, $val, $c ) = @_;

    $c ||= '';

    die( "unknown field: $param\n" )
      unless defined $field{$param};

    $field{$param}{format} //= '%s';

    printf $fh (
        "%-32s%s\n",
        sprintf(
            "%d,%d,$field{$param}{format}",
            $rec_type, $field{$param}{id}, $val
        ),
        $field{$param}{comment} . $c
    );
}

