-- --8<--8<--8<--8<--
--
-- Copyright (C) 2010 Smithsonian Astrophysical Observatory
--
-- This file is part of raygen
--
-- raygen is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--
-- -->8-->8-->8-->8--

-- extent generator call validation

local raygen 	= require( 'saotrace.raygen' )
local idist 	= require( 'saotrace.raygen.validate.idist' )
local vobj 	= require( 'saotrace.raygen.validate').vobj
local units 	= require( 'saotrace.raygen.units' )
local intrinsic = require( 'saotrace.raygen.intrinsic.extent' )


local _M = {}

--------------------------
-- this holds the validation specifications for the generators.  it
-- cannot be local because intrinsic uses it to create mock functions

_M.vspec = {}
local vspec = _M.vspec

--------------------------
-- Object constructors; don't call directly;
-- use the type specific wrappers created below


local Base = {}

function Base:new( )

   local obj = {}
   setmetatable(obj, self)
   self.__index = self
   return obj

end

function Base:register( )

  return intrinsic[self.type](self.args)

end

function Base:isa( class )

  return class == 'saotrace.raygen.extent'

end

function _M.default( args )

   return _M.point( args )

end

--------------------------

-- keep track of options for each specification

local options = {}

---------------------------
-- generic source position arguments
-- for now we just do angular ones.

local position = {

   optional = true,
   vtable = {

      theta = { default = { 0, units.angular.radian }, type = 'zpos_angle_arcminute'    },
      phi   = { default = { 0, units.angular.radian }, type = 'angle_degree'    },
      z     = { default = { -1e30, units.linear.mm } , type = 'distance' }

   }
}


--------------------------

vspec.disk = {
   name   = { type = 'string' },
   radius = { type = 'ang_lin_distance' },
   idist  = { type = 'idist', optional = true },
   position = position,
}


--------------------------

vspec.rect = {
   name   = { type = 'string' },
   width  = { type = 'ang_lin_distance' },
   height = { type = 'ang_lin_distance' },
   rotate  = { type = 'angle_degree', default = { 0, units.angular.radian } },
   idist  = { type = 'idist', optional = true },
   position = position,
}

--------------------------

local image_pixsz = { type = 'posnum', optional = true }
local image_pixsz_tbl = { optional = 'true',
		    vtable = { x = { type = 'posnum' },
			       y = { type = 'posnum' } } }

vspec.image = {

   name = { type = 'string' },
   position = position,

   rotate  = { type = 'angle_degree', default = { 0, units.angular.radian } },

   format = {
      optional = true,

      enum = { 'ascii',
	       'double',
	       'float',
	       'fits',
	       'pbm',
	       'pgm' }
   },

   file = {
      type = 'string'
   },

   clip = {
      type = 'zposnum',
      default = 0
   },

   flip = {
      optional = true,
      enum = { 'x', 'y', 'xy', 'none' },
      default = 'none'
   },

   pixsz = {
      default = { 0.5, units.angular.arcsecond },
      type = {

	 xy = {
	    optional = true,
	    type = 'pos_angle_arcsecond',
	 },

	 x_y = {
	    optional = true,
	    vtable = {
	       x = { type = 'pos_angle_arcsecond' },
	       y = { type = 'pos_angle_arcsecond' },
	    }
	 }

      }
   },

   idist = {
      optional = true,
      default_is_nil = true,
      vtable = idist.vtable,
   },

   size = {
      optional = true,
      default_is_nil = true,
      vtable = {
	 x = { type = 'posnum' },
	 y = { type = 'posnum' },
      }
   },

   point_idx = {
      optional = true,
      default_is_nil = true,
      vtable = {
	 x = { type = 'number' },
	 y = { type = 'number' },
      },
      excludes = 'point_wcs'
   },

   point_wcs = {
      optional = true,
      default_is_nil = true,
      vtable = {
	 x = { type = 'number' },
	 y = { type = 'number' },
      },
      excludes = 'point_idx',
      -- want exactly one of these
      one_of = { 'ref_wcs', 'ref_idx' },
   },

   ref_wcs = {
      optional = true,
      default_is_nil = true,
      vtable = {
	 x = { type = 'number' },
	 y = { type = 'number' },
      },
      requires = 'point_wcs',
      excludes = 'ref_idx',
   },

   ref_idx = {
      optional = true,
      default_is_nil = true,
      vtable = {
	 x = { type = 'number' },
	 y = { type = 'number' },
      },
      requires = 'point_wcs',
      excludes = 'ref_wcs'
   },
}

---------------------------

vspec.point = {

   name = { type = 'string' },
   position = position,

}


-- create wrappers around compiled functions

for gen, specs in pairs( vspec ) do

   _M[gen] = function (...)
   	     	  local obj = Base:new()
		  obj.type = gen
		  local ok, args

		  -- if there are options for this generator clone the
		  -- raygen validation object at run time (not compile time)
		  -- to track changes to it.
		  if options[gen] then
		     local vobj = vobj:new()
		     vobj:setopts( options[gen] )
		     ok, args = vobj:validate( specs, ... )
		  else
		     ok, args = vobj:validate( specs, ... )
		  end
		  if ok then
		     obj.args = args;
		     return obj
		  else
		     return error( args, 2 )
		  end
	       end
end

return _M
