// --8<--8<--8<--8<--
//
// Copyright (C) 2006 Smithsonian Astrophysical Observatory
//
// This file is part of BinaryPipe
//
// BinaryPipe is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// BinaryPipe is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the 
//       Free Software Foundation, Inc. 
//       51 Franklin Street, Fifth Floor
//       Boston, MA  02110-1301, USA
//
// -->8-->8-->8-->8--

#define BinaryPipe_CC
#include <BinaryPipe/BinaryPipe.h>

template <class Type>
BinaryPipe<Type>::~BinaryPipe( ) {

  if ( instream ) {
    bpipe_delete( instream );
    instream = NULL;
  }

  if ( photon ) {
    delete photon;
    photon = NULL;
  }

}

template <class Type>
BinaryPipe<Type>::BinaryPipe( const char* input, const char* output,
			      HeaderField hdrf[], DataPacket packet[] ) 
  : instream( NULL ), core( NULL ), 
    outstream( NULL ), photon( NULL ) {
  
  try {

    init_BinaryPipe( input, output, hdrf, packet );

  } catch( BinaryPipeException& bpe ) {

    throw;

  }

}

template <class Type>
void BinaryPipe<Type>::init_BinaryPipe( const char* input,
					const char* output,
					HeaderField hdrf[], 
					DataPacket packet[] ) {

    char msg[256];

    if ( hdrf || packet ) {

      if ( NULL == output ) {
	const char* format =
	  "BinaryPipe<T>::init_BinaryPipe( %s, NULL, NULL, NULL ) : "
	  "The output stream must be specified if a header field or "
	  "a data packet is to be added\n";
	sprintf( msg, format, input );
	throw BinaryPipeException( msg );
      }

    }

    instream = bpipe_new( );
    if ( NULL == instream ) {
      const char* format =
	"BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], DataPacket[] )"
	" : NULL == instream (%s)\n";
      sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
      throw BinaryPipeException( msg );
    }

    if ( bpipe_input( instream, (char*) input ) ) {
      const char* format =
	"BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], DataPacket[] )"
	" : bpipe_input (%s)\n";
      sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
      throw BinaryPipeException( msg );
    }

    // To add the header:
    if ( output && hdrf ) {

      for ( int ii = 0; NULL != packet[ ii ].name; ii++ ) {

	if ( bpipe_hdrf_add( instream, (char*) hdrf[ ii ].name,
			     hdrf[ ii ].type, hdrf[ ii ].matrix,
			     hdrf[ ii ].data, hdrf[ ii ].copy ) ) {
	  
	  const char* format =
	    "BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], "
	    "DataPacket[] ) : bpipe_hdrf_add (%s)\n";
	  sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
 	  throw BinaryPipeException( msg );
	}

      }

    }
  
    if ( output ) {

      outstream = bpipe_output( instream, (char*) output );
      if ( NULL == outstream ) {
	const char* format =
	  "BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], "
	  "DataPacket[] ) : bpipe_output (%s)\n";
	sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
	throw BinaryPipeException( msg );
      }

    }

    if ( output && packet ) {

      for ( int ii = 0; NULL != packet[ ii ].name; ii++ ) {

	if ( -1 == bpipe_dpktf_add( instream, 
				    (char*) packet[ ii ].name,
				    packet[ ii ].type,
				    packet[ ii ].matrix ) ) {

	  const char* format =
	    "BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], "
	    "DataPacket[] ) : bpipe_dpkt_add (%s)\n";
	  sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
	  throw BinaryPipeException( msg );

	}
	
      }

    }

    // map data packet fields
    core = bpipe_map_alloc( instream, 1, NULL );
    if ( NULL == core ) {
      const char* format =
	"BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], "
	"DataPacket[] ) : NULL == core (%s)\n";
      sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
      throw BinaryPipeException( msg );
    }

    if ( output )
      if ( bpipe_write_hdr( instream ) ) {
	const char* format =
	  "BinaryPipe<T>::init_BinaryPipe( %s, %s, HeaderField[], "
	  "DataPacket[] ) : bpipe_write_hdr (%s)\n";
	sprintf( msg, format, input, output, bpipe_strerror( bpipe_errno ) );
	throw BinaryPipeException( msg );
      }

    try {

      photon = new Type( instream, core );

    } catch( BinaryPipeException& bpe ) {

      throw;

    }

}

