#ifndef TraceFct_h_INCLUDED
#define TraceFct_h_INCLUDED

// --8<--8<--8<--8<--
//
// Copyright (C) 2006-2012 Smithsonian Astrophysical Observatory
//
// This file is part of tracefctxx
//
// tracefctxx is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or (at
// your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -->8-->8-->8-->8--

#include <iosfwd>
#include <list>
#include <cstdlib>
#include <string>

#include <Exception/Exception.h>

/*!
  \file TraceFct.h
  The class definitions
*/


/*!
  \def _tf_assert
  \internal
*/

# define _tf_assert(level,ex)    \
  do									\
  {									\
    if (!(ex))								\
      TraceFct::exit( level, "Assertion failed: file\"%s\", line %d\n%s\n",	\
	    __FILE__, __LINE__, #ex);					\
  } while(0)

/*! \def TF_ASSERT_LEVEL

   \brief This controls the level-based assertion macros.  There are
   five levels; this macro should be assigned an integer corresponding
   to the maximum level for which assertions should be enabled.  All
   assertions at levels less than or equal to this will be enabled.
   It is not defined by default.

 */

#ifdef DOXYGEN
#  define TF_ASSERT_LEVEL
#endif


/*! \def tf_assert(assertion)
    \brief if \a assertion is not true, generate a message and exit
    with an error code of \c 990.

*/

# define tf_assert(ex)    _tf_assert(990,ex)

#ifdef TF_ASSERT_LEVEL
#  if TF_ASSERT_LEVEL > 0

/*! \def tf_assert1(assertion)
    \brief if \a assertion is not true and  \ref TF_ASSERT_LEVEL is greater
    or equal to 1, generate a message and exit
    with an error code of \c 991.
*/

#    define tf_assert1(ex)     _tf_assert(991,ex)
#    if TF_ASSERT_LEVEL > 1

/*! \def tf_assert2(assertion)
    \brief if \a assertion is not true and \ref TF_ASSERT_LEVEL is greater
    or equal to than 1, generate a message and exit
    with an error code of \c 992.
*/

#      define tf_assert2(ex)     _tf_assert(992,ex)
#      if TF_ASSERT_LEVEL > 2

/*! \def tf_assert3(assertion)
    \brief if \a assertion is not true and \ref TF_ASSERT_LEVEL is greater
    or equal to than 3, generate a message and exit
    with an error code of \c 993.
*/

#        define tf_assert3(ex)     _tf_assert(993,ex)
#        if TF_ASSERT_LEVEL > 3

/*! \def tf_assert4(assertion)
    \brief if \a assertion is not true and \ref TF_ASSERT_LEVEL is greater
    or equal to than 4, generate a message and exit
    with an error code of \c 994.
*/

#          define tf_assert4(ex)     _tf_assert(994,ex)
#          if TF_ASSERT_LEVEL > 4

/*! \def tf_assert5(assertion)
    \brief if \a assertion is not true and \ref TF_ASSERT_LEVEL is greater
    or equal to than 5, generate a message and exit
    with an error code of \c 995.
*/


#            define tf_assert5(ex)    _tf_assert(995,ex)
#          endif  /* TF_ASSERT_LEVEL > 4 */
#        endif  /* TF_ASSERT_LEVEL > 3 */
#      endif  /* TF_ASSERT_LEVEL > 2 */
#    endif  /* TF_ASSERT_LEVEL > 1 */
#  endif  /* TF_ASSERT_LEVEL > 0 */
#endif  /* TF_ASSERT_LEVEL     */

#ifndef tf_assert1
#  define tf_assert1(ex)
#endif

#ifndef tf_assert2
#  define tf_assert2(ex)
#endif

#ifndef tf_assert3
#  define tf_assert3(ex)
#endif

#ifndef tf_assert4
#  define tf_assert4(ex)
#endif

#ifndef tf_assert5
#  define tf_assert5(ex)
#endif


/*!
  The TraceFct Class

*/
class TraceFct {

 public:
  TraceFct( const char* name );
  TraceFct( std::string& name );
  TraceFct( std::string name, bool print_it, int num_fct_to_print );
  ~TraceFct( );

  static void exit (int exit_code, const char *format, ...);
  static void exit (int exit_code, Exception& ex );

  static void exit (int exit_code, const std::string& msg)
    { exit( exit_code, msg.c_str() ); };


  static void die (const char *format, ...);

  static void die ( Exception& ex )
    { exit( EXIT_FAILURE, ex ); }

  static void die ( const std::string& msg)
    { exit( EXIT_FAILURE, msg ); };

  static void message (const char *format, ...);
  static void message ( Exception& ex )
    { print( false, ex ); }
  static void message (const std::string& msg)
    { message( msg.c_str() ); }

  static void vmessage (const char *format, va_list args);

  static void println ( Exception& ex )
    { print( true, ex ); }
  static void println (const std::string& msg)
    { print( true, msg.c_str() ); }
  static void println (const char* msg)
    { print( true, msg ); }

  static void dump_stack (void);
  static void open( const std::string& filename );
  static void close(void);

 private:
  static bool print_upon_enter_and_exit;
  static int stack_level_to_print;
  static std::string prefix;

  // a scratch buffer; need this preallocated if have run out
  // memory later on
  static char outbuf[8192];
  static std::string progname;
  static std::ostream* ostr;
  static std::list<std::string> function_stack;
  
  static char*  init_prefix( void );

  static void vexit_print(const char* format, va_list args );
  static void vprint(bool print_nl, const char *format, va_list args);
  static void print(bool print_nl, const char *format, ... );
  static void print(bool print_nl, Exception& ex );

};




#endif  /* ! tracefct_h_INCLUDED */
