#ifndef EXCEPTION_H
#define EXCEPTION_H

// --8<--8<--8<--8<--
//
// Copyright (C) 2006 Smithsonian Astrophysical Observatory
//
// This file is part of Exception
//
// Exception is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// Exception is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the 
//       Free Software Foundation, Inc. 
//       51 Franklin Street, Fifth Floor
//       Boston, MA  02110-1301, USA
//
// -->8-->8-->8-->8--

#include <stdarg.h>
#include <iostream>
#include <deque>
#include <string>
#include <stdexcept>

/**
 *  The class Exception, all the subclasses as defined by the user,
 *  are a form of conditions that a reasonable application might want to 
 *  catch. 
 */

class Exception : public std::runtime_error {

  /**
   * Prints this and its backtrace to the specified output stream.
   */

  friend std::ostream& operator << ( std::ostream& os, Exception& a ) {
    a.print( os );
    return os;
  }

  /**
   * Prints this and its backtrace to the specified output stream.
   */

  friend std::ostream& operator << ( std::ostream& os, Exception* a ) {
    os << *a;
    return os;
  }

public:

  // GCC (and Stroustrup) require a virtual function may be overridden */
  // only by a function that has an exception-specification AT LEAST
  // as restrictive as its own...
  virtual ~Exception( ) { }

  /**
   * Constructs an Exception with no specified detail message.
   */

  Exception( ) : runtime_error( "" ) { }

  Exception( const Exception& e ) : runtime_error( e.what( ) ) {
    //cerr << "The copy constructor Exception( const Exception& ) was called\n";
    this->operator=( e );
  }

  /**
   * Constructs an Exception with a specified detail message.
   */
  Exception( const std::string& arg ) : runtime_error( arg ) { 
    // cerr << "Exception( const string& ) was called\n";
    set_message( arg );
  }

  Exception& operator= ( const Exception& rhs );

  /**
   * Get the iterator pointing to the beginning of the dequeue.
   */
  std::deque<std::string>::const_iterator begin( ) const {
    return exception_queue.begin( );
  }

  /**
   * Get the iterator pointing to the end of the dequeue.
   */
  std::deque<std::string>::const_iterator end( ) const {
    return exception_queue.end( );
  }

  /**
   * Get the most current message
   */
  std::string get_message( void ) const {
    return exception_queue.empty() ? "" : exception_queue[ 0 ];
  }

  /**
   * Add a message.
   */
  void set_message( const std::string& msg );

  void set_rethrow_message( const std::string& file, const int linenum );

  const char* what() const noexcept;

  void update_what( );

protected:

  std::deque<std::string> exception_queue;

  /**
   * Prints this and its backtrace to the specified output stream.
   */
  virtual void print( std::ostream& os=std::cerr ) const;

  std::string _what;

};

#define RETHROWME( arg ) {arg.set_rethrow_message( __FILE__, __LINE__ ); throw arg;}

/** \example test.cc
 * This is an example of how to use the Test class.
 * More details about this example.
 */

#endif
