#ifndef RDBColumn_h
#define RDBColumn_h

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of RDB
 *
 * RDB is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * RDB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <cerrno>
#include <cstdio>
#include <cstring>
#include <float.h>
#include <iomanip>
#include <iostream>
#include <limits>
#include <sstream>
#include <string>

#include <rdbxx/RDBErr.h>

/// Provides interface for general column related methods.
class RDBColumn {
  /// @name Stream insertion and extraction operators.
  //@{
  /// Read column from input stream.
  friend std::istream& operator>>( std::istream& is, RDBColumn& col );
  /// Read column from input stream.
  friend std::istream& operator>>( std::istream& is, RDBColumn* col );
  /// Write column to output stream.
  friend std::ostream& operator<<( std::ostream& os, const RDBColumn& col );
  /// Write column to output stream.
  friend std::ostream& operator<<( std::ostream& os, const RDBColumn* col );
  //@}
  
 public:
  /// @name Enumerations for column definitions and error conditions.
  //@{
  /// Acceptable column justifications.
  enum Just { LEFT='<', RIGHT='>' };
  /// Acceptable column types.
  enum Type { MONTH='M', NUMERIC='N', STRING='S' };
  /// Possible error conditions.
  enum Err  { NOERR=0, LOSTPRECISION=EDOM, NONNUMERIC=EINVAL, OUTOFRANGE=ERANGE, NODATA=ENODATA };
  enum Status { CMOG=0x01, CEOG=0x02, CBOG=0x04 };
  //@}
 
  
  /// @name Constructing, destructing, and initializing RDBColumn objects.
  //@{
  /// Assigns name and definition to RDBColumn object.
  RDBColumn( const std::string& name="", const std::string& def="" );
  /// Copies RDBColumn object.
  RDBColumn( const RDBColumn& col );
  /// Deletes resources allocated by RDBColumn object.
  virtual ~RDBColumn( void );

  /// Copies RDBColumn object.
  RDBColumn& operator=( const RDBColumn& col );
  /// Assigns data to RDBColumn object's _data member, converting as necessary.
  virtual RDBColumn& operator=( const double& data ) = 0;
  /// Assigns data to RDBColumn object's _data member, converting as necessary.
  virtual RDBColumn& operator=( const long& data ) = 0;
  /// Assigns data to RDBColumn object's _data member, converting as necessary.
  virtual RDBColumn& operator=( const std::string& data ) = 0;
  //@}

  /// @name Auto-indexing control methods.
  //@{
  /// Increments index to the RDBColumn's data elements.
  virtual void advanceIdx( void ) = 0;
  /// Rewinds index to the RDBColumn's data elements.
  virtual void rewind( void ) = 0;
  //@}

  /// @name Group information ("break" column) methods.
  //@{
  /// Turn on/off group tracking for this column object.
  virtual void setGroup( bool group );
  /// Returns group status, RBOG if at beginning of a group, REOG if
  /// at ned of a group, or REOL if in the middle of a group.
  bool getGroup( void ) const;
  /// Sets the group value to the current data value.
  virtual void setGroupValue( void ) = 0;
  /// Returns the group status of this column object.
  virtual int newGroup( void ) = 0;
  //@}
  
  /// @name Data member initializers.
  //@{
  /// Sets the name.
  void setName( const std::string& name );
  /// Sets the definition.
  void setDef( const std::string& def );
  /// Sets the width.
  void setWidth( const long width );
  /// Sets the type.
  void setType( const RDBColumn::Type type );
  /// Sets the justification.
  void setJust( const RDBColumn::Just just );
  /// Sets the description.
  void setDesc( const std::string& desc );
  /// Sets the precision for numeric output and numeric to string conversion.
  void setPrecision( const int precision );
  /// Sets the excpeption throwing behavior.
  void setThrow( const bool t=true );
  /// Sets the error status.
  void setErrNo( const int no=0 );  
  /// Sets the data value, converting as necessary.
  virtual bool setData( const double& data ) = 0;
  /// Sets the data value, converting as necessary.
  virtual bool setData( const long& data ) = 0;
  /// Sets the data value, converting as necessary.
  virtual bool setData( const std::string& data ) = 0;
  //@}

  /// @name Methods to map RDBColumn's data to user-supplied memory.
  //@{
  /// Maps data to user-supplied memory, if possible.
  virtual void mapData( double data[], const size_t nelems=1 );
  /// Maps data to user-supplied memory, if possible.
  virtual void mapData( long data[], const size_t nelems=1 );
  /// Maps data to user-supplied memory, if possible.
  virtual void mapData( std::string data[], const size_t nelems=1 );
  //@}

  /// @name Data member accessors.
  //@{
  /// Returns the name.
  std::string getName( void ) const;
  /// Returns the definition.
  std::string getDef( void ); // Non-constant, because it has to
                        //   reconstruct the definition if any field
                        //   has changed... 
  /// Returns the width.
  long getWidth( void ) const;
  /// Returns the type.
  RDBColumn::Type getType( void ) const;
  /// Returns the justification.
  RDBColumn::Just getJust( void ) const;
  /// Returns the description.
  std::string getDesc( void ) const;
  /// Returns the precision.
  int getPrecision( void ) const;
  /// Returns the state of the exception throwing behavior.
  bool getThrow( void ) const;
  /// Returns a brief description of the error condition.
  char* getErr( void ) const;
  /// Returns the error status.
  int getErrNo( void ) const;
  /// Returns a pointer to the current data element.
  virtual void* getData( void ) = 0;
  /// Returns the value of the current data element, converting if necessary.
  virtual bool getData( double& data ) = 0;
  /// Returns the value of the current data element, converting if necessary.
  virtual bool getData( long& data ) = 0;
  /// Returns the value of the current data element, converting if necessary.
  virtual bool getData( std::string& data ) = 0;
  /// Returns the value of the current data element, converting if necessary.
  virtual double getDataDouble( void ) = 0;
  /// Returns the value of the current data element, converting if necessary.
  virtual long getDataLong( void ) = 0;
  /// Returns the value of the current data element, converting if necessary.
  virtual std::string getDataString( void ) = 0;
  //@}
  
 protected:
  /// Used to converted data based on user requests.
  void convert( const double& idata, double& odata ); 
  /// Used to converted data based on user requests.
  void convert( const double& idata, long& odata );
  /// Used to converted data based on user requests.
  void convert( const double& idata, std::string& odata );
  /// Used to converted data based on user requests.
  void convert( const long& idata, double& odata );
  /// Used to converted data based on user requests.
  void convert( const long& idata, long& odata );
  /// Used to converted data based on user requests.
  void convert( const long& idata, std::string& odata );
  /// Used to converted data based on user requests.
  void convert( const std::string& idata, double& odata ); 
  /// Used to converted data based on user requests.
  void convert( const std::string& idata, long& odata ); 
  /// Used to converted data based on user requests.
  void convert( const std::string& idata, std::string& odata );

  /// Called by the stream insertion operator.
  virtual std::istream& read( std::istream& is ) = 0;
  /// Overridden in the subclass of this datatype.
  virtual std::istream& extract( std::istream& is, double& data );
  /// Overridden in the subclass of this datatype.
  virtual std::istream& extract( std::istream& is, long& data );
  /// Overridden in the subclass of this datatype.
  virtual std::istream& extract( std::istream& is, std::string& data );
  /// Called by the stream extraction operator.
  virtual std::ostream& write( std::ostream& os ) const = 0;
  /// Overridden in the subclass of this datatype.
  virtual std::ostream& insert( std::ostream& os, double& data ) const;
  /// Overridden in the subclass of this datatype.
  virtual std::ostream& insert( std::ostream& os, long& data ) const;
  /// Overridden in the subclass of this datatype.
  virtual std::ostream& insert( std::ostream& os, std::string& data ) const;

  /// Name.
  std::string _name;
  /// Definition.
  std::string _def;
  /// Width.
  long _width;
  /// Data type.
  RDBColumn::Type _type;
  /// Justification.
  RDBColumn::Just _just;
  /// Description.
  std::string _desc;
  /// Indicates state for the definition field.
  bool _changed;
  
  /// State of the exception throwing behavior.
  bool _throw;
  /// Error state.
  int _errno;
  /// Precision used for stream output or numeric to std::string conversion.
  int _precision;
  /// Used for numeric to string conversion.
  std::stringstream _strstrm;
  /// This is a group column.
  bool _group;
  /// Group been initialized.
  bool _initgroup;
  
};

#endif
