#ifndef RDBComment_h
#define RDBComment_h

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of RDB
 *
 * RDB is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * RDB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <iostream>
#include <string>
#include <rdbxx/RDBErr.h>

/// Provides interface for manipulating RDB comments.
class RDBComment {
  /// @name Stream insertion and extraction operators.
  //@{
  /// Read comment from input stream.
  friend std::istream& operator>>( std::istream& is, RDBComment& rdbcomment );
  /// Write comment to output stream.
  friend std::ostream& operator<<( std::ostream& os, const RDBComment& rdbcomment );
  //@}
  
 public:
  /// @name Constructing, destructing, and initializing RDBComment objects.
  //@{
  /// Parses comment string for RDB comment structure.
  RDBComment( const std::string& comment="" );
  /// Copy RDBComment object.
  RDBComment( const RDBComment& rdbcomment );
  /// Destructor has nothing to do.
  ~RDBComment( void );

  /// Copy RDBComment object.
  const RDBComment& operator=( const RDBComment& rdbcomment );
  /// Copy string to RDBComment object.
  const RDBComment& operator=( const std::string& rdbcomment );
  //@}

  /// @name Data member initializers.
  //@{
  /// Parses comment string for RDB comment elements.
  void setComment( const std::string& comment ); 
  /// Parses string for RDB comment elements.
  void setCommentText( const std::string& comment ); 
  /// Set just the comment keyword.
  void setKeyword( const std::string& keyword );
  /// Set just the comment value.
  void setValue( const std::string& value );
  //@}

  /// @name Data member accessors.
  //@{
  /// Return the full comment.
  std::string getComment( void ) const;
  /// Return the full comment text.
  std::string getCommentText( void ) const;
  /// Return the keyword, if any.
  std::string getKeyword( void ) const;
  /// Return the keyword's value, if any.
  std::string getValue( void ) const;
  //@}
  
private:
  /// Entire comment line.
  std::string _comment;
  /// Keyword component, if any.
  std::string _keyword;
  /// Keyword's value component, if any.
  std::string _value;
  
};

#endif
