#ifndef rl_DielectricLayer_h_INCLUDED
#define rl_DielectricLayer_h_INCLUDED

// File:   rl_DielectricLayer.h
// Author: Terry Gaetz
 
/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * encapsulate the dielectric layer properties:
 *
 * History
 *--------
 * 1.0.1 2004-Dec-07  tjg  simplify/remove include guards
 * 0.0.0 1998-Jun-24  tjg  original version
 */

#include <cstdlib>                            // size_t   
#include <rl_raylib/rl_Traits.h>              // rl_Traits::EInterpMode complex
#include <rl_raylib/rl_Exception.h>           // rl_Exception
#include <rl_raylib/rl_DielectricData.h>      // rl_DielectricData
#include <rl_raylib/rl_ReflectionCoefPOD.h>   // rl_ReflectionCoefPOD
#include <rl_raylib/rl_TransmissionCoefPOD.h> // rl_ReflectionCoefPOD
#include <rl_raylib/rl_Exception.h>           // rl_Exception

//########################################################################
// rl_DielectricLayer
//########################################################################
//
/** 
 * \class rl_DielectricLayer
 *
 * A class encapsulating the multilayer reflection of a ray.
 * Given the energy and the sine of the graze angle, the 
 * reflection coefficient is evaluated.
 */

class rl_DielectricLayer
{
public:

  /// complex type
  typedef rl_Traits::complex    complex;

  /// roughness type
  typedef rl_Traits::ERoughType ERoughType;

private:

  rl_DielectricLayer& operator=( rl_DielectricLayer const& );  /// PROHIBITED

  /// The layer name
  char*                   name_;

  /// Descriptor for the dielectric data for the layer
  rl_DielectricData       diel_info_;

  /// The thickness of the layer (in Angstrom). 
  double                  thickness_;

  /// The dimensionless thickness of layer, i.e.
  /// \f[
  ///   \mathtt{zcoat\_} \equiv {4 \pi d \over \lambda}
  /// \f]
  /// where #thickness_\f$ = d\f$ is the physical thickness
  /// of the layer (in Angstrom) and #lambda_\f$ = \lambda\f$
  /// is the X-ray wavelength (in Angstrom).
  double                  zcoat_;

  /// The real part of the dielectric decrement
  double                  alpha_;

  /// The imaginary part of the dielectric decrement
  double                  gamma_;

  /// The layer roughness type
  ERoughType              rtype_;

  /// The roughness parameter (Angstroms)
  double                  srough_;

  /// X-ray wavelength (Angstrom, in vacuum)
  double                  lambda_;

  /// Reflectivity propagator
  complex                 prop_;

  /// This is the component of the wave vector perpendicular to the 
  /// interface: \f$k_\mathit{perp} / k_0\f$, 
  /// where \f$k_0\f$ is the vacuum wavenumber, i.e., the magnitude of 
  /// the wave vector in vacuum.
  ///
  /// The boundary condition at the interface requires that the component
  /// of the wave vector parallel to the interface is the same for each
  /// layer.  The perpendicular component is then
  /// \f[
  ///    k_{\perp,j} = \sqrt{ \sin^2 \phi_0 + \Delta \epsilon_\mathit{j} }
  /// \f]
  /// where \f$\phi_0\f$ is the graze angle in the vacuum and
  /// \f$\Delta\epsilon_\mathit{j}\f$ is the dielectric decrement, i.e.,
  /// complex\f$( -\alpha_\mathit{j}, -\gamma_\mathit{j} )\f$.
  /// Note that in obtaining \f$k_{\perp,j}\f$ we take a complex square
  /// root.  The imaginary part of the principal root is obtained
  /// by multiplying \f$b\f$ by the sign of #gamma_.
  complex                 kt_perp_; 

  /// The complex phase factor: 
  ///  \f${2\pi\sigma_\mathit{rough}\over\lambda}\f$ 
  /// where \f$\sigma_\mathit{rough} \equiv \mathtt{srough\_}\f$
  /// and \f$\lambda \equiv \mathtt{lambda\_}\f$ is the wavelength
  /// in vacuum.
  double                  phase_factor_;  // 2\pi srough_ / lambda_

  /// The complex reflection coefficient.
  rl_ReflectionCoefPOD    rflcoef_;

  /// The complex transmission coefficient for ray hitting (from above) 
  /// the interface between this layer and the layer above
  rl_TransmissionCoefPOD  t_ij_;

  /// complex transmission coefficient for ray hitting (from below)
  /// the interface between this layer and the layer above
  rl_TransmissionCoefPOD  t_ji_;

  /// conservation quantity for ideal planar dielectric
  complex                 tij_tji_para_;

  /// conservation quantity for ideal planar dielectric
  complex                 tij_tji_perp_;

  /// conservation quantity for ideal planar dielectric
  complex                 r_para_;

  /// conservation quantity for ideal planar dielectric
  complex                 r_perp_;

  /// conservation quantity for ideal planar dielectric
  complex                 r2_para_;

  /// conservation quantity for ideal planar dielectric
  complex                 r2_perp_;

  /// conservation quantity for ideal planar dielectric
  complex                 r2_tij_tji_para_;

  /// conservation quantity for ideal planar dielectric
  complex                 r2_tij_tji_perp_;

  complex                 sqrt_epsratio_;
  complex                 eps_epsupper_;
  complex                 sqrt_eps_epsupper_;

  /// True if this layer is the substrate; otherwise False.
  rl_Traits::Bool         is_substrate_;

public:

  /** 
   * Destructor
   */
 ~rl_DielectricLayer();

  /** 
   * Constructor.  By default, constructs a VACUUM layer.
   * Use init method to initialize for conditions other than vacuum.
   *
   * @param layer_name optional string naming the layer.
   */
  rl_DielectricLayer( char const layer_name[] = 0 );

  /** 
   * DEEP Copy constructor
   *
   * @param other rl_DielectricLayer to be copied.
   */
  rl_DielectricLayer( rl_DielectricLayer const& other );

  /** 
   * Constructor.
   *
   * @param  diel     array of dielectric decrement information.
   * @param  ndielpts number of points in diel array.
   * @param  layer_thickness layer thickness (Angstroms).
   * @param  roughness       layer ``roughness'' (Angstroms).
   * @param  roughness_type  interlayer grading ``roughness'' type.
   * @param  interp_mode     type of optical constant interpolation
   *           to be done.
   * @param  bulkdensity     relative bulk density factor; 1 for full bulk 
   *                         density.
   * @param  layer_name      string describing the layer composition.
   * @param  is_substrate    boolean:  True if this is the substrate layer.
   */
  rl_DielectricLayer( rl_Traits::rl_DielectricPOD const* 
                                               diel,
                      std::size_t              ndielpts,
                      double                   layer_thickness,
                      double                   roughness,
                      rl_Traits::ERoughType    roughness_type,
                      rl_Traits::EInterpMode   interp_mode,
                      double                   bulkdensity,
                      char const*              layer_name,
                      rl_Traits::Bool          is_substrate = rl_Traits::False 
                    );

  /** 
   * Initializer
   *
   * @param diel     array of dielectric decrement information.
   * @param ndielpts number of points in diel array.
   * @param layer_thickness layer thickness (Angstroms).
   * @param roughness       interlayer grading "roughness" (Angstrom). 
   * @param roughness_type  interlayer grading "roughness" type. 
   * @param interp_mode     type of optical constant interpolation
   *           to be done.
   * @param bulkdensity     relative bulk density factor; 1 for full bulk
   *           density.
   * @param layer_name      string describing the layer composition.
   * @param is_substrate    boolean:  True if this is the substrate layer.
   */

  void init( rl_Traits::rl_DielectricPOD const* diel,
             std::size_t                        ndielpts,
             double                             layer_thickness,
             double                             roughness,
             rl_Traits::ERoughType              roughness_type,
             rl_Traits::EInterpMode             interp_mode,
             double                             bulkdensity,
             char const*                        layer_name,
             rl_Traits::Bool                    is_substrate = rl_Traits::False 
           );

  /** 
   * Set up layer state for given energy and graze angle.
   *
   * @param energy energy (keV).
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *       	and the top surface of the multilayer.
   */
  int setup_for( double energy, double sinphi );

  /**
   * Compute reflectivity for a stack of num layers.
   *
   * PRECONDITION:
   *    The layers must be pre-initialized with setup_for
   *    and the reflection coefficients evaluated with
   *    reflect_amp.
   *
   * @param layer array of layers.
   * @param num   number of layers in the array.
   */
  void reflect_nlayer( rl_DielectricLayer layer[],
                       int                num     );

  /**
   * Compute reflection amplitude for the interface between this layer 
   * and the layer immediately above it.
   * sinphi is the graze angle at the topmost layer (i.e., in vacuum).
   * Layer "layer" is assumed to be adjacent to this layer, with "layer"
   * nearer the vacuum and this layer nearer the substrate.  The reflection 
   * coefficient for this layer is updated.
   *
   *  PRECONDITION:
   *    The layers must be pre-initialized with setup_for
   *
   * @param layer adjacent layer above this layer, where above means
   *        closer to the vacuum.
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *        and the top surface of the multilayer.
   */
  void reflect_amp( rl_DielectricLayer const& layer,
                    double                    sinphi );

  /**
   * The propagator for this layer.
   *
   * @return propagator for this layer.
   */
  complex const& propagator() const;

  /**
   * Returns this layer's dielectric decrement (real part).
   *
   * @return this layer's dielectric decrement (real part).
   */
  double alpha() const;

  /**
   * Returns this layer's dielectric decrement (imag part).
   *
   * @return this layer's dielectric decrement (imag part).
   */
  double gamma() const;

  /**
   * Returns the roughness parameter of the upper surface of this layer.
   *
   * @return the roughness parameter of the upper surface of this layer.
   */
  double roughness() const;

  /**
   * Returns the roughness type of the upper surface of this layer.
   *
   * @return the roughness type of the upper surface of this layer.
   */
  ERoughType roughness_type() const;

  /**
   * Returns this layer's complex reflection coefficient.
   *
   * \return this layer's complex reflection coefficient.
   */
  rl_ReflectionCoefPOD const& reflection_coef() const;

  /**
   * Returns: this layer's reflectivity.
   *
   * @param polarization_factor - polarization factor; it must be
   *        a value between -1 and 1.
   *
   * The polarization factor is related to parallel (p) and
   * perpendicular (s) polarization by:
   *   polarization_factor \f$ = (I_\mathit{perp} - I_\mathit{para}) 
   *                      / (I_\mathit{perp} + I_\mathit{para})\f$
   * or
   *   polarization_factor \f$ = (I_\mathit{s} - I_\mathit{p}) 
   *                      / (I_\mathit{s} + I_\mathit{p})\f$
   *  where \f$I_\mathit{perp} and I_\mathit{para}\f$ are the perpendicular
   *  and parallel E-field \em intensities, respectively.  Thus,
   * - -1: pure parallel (p) polarization.
   * -  0: completely unpolarized.
   * - +1: pure perpendicular (s) polarization.
   *
   * \return this layer's reflectivity
   */
  double reflectivity( double polarization_factor = 0.0 ) const;

  /**
   * @return rl_Traits::True if this layer is the substrate, 
   *    rl_Traits::False otherwise.
   */
  rl_Traits::Bool is_substrate() const;

  /**
   * @return rl_Traits::True if this layer is the vacuum, 
   *    rl_Traits::False otherwise.
   */
  rl_Traits::Bool is_vacuum() const;

  /**
   * @return the name of this layer as a character string.
   */
  char const* layer_name() const;

  /**
   * @return the minimum energy allowed for this layer.
   */
  double energy_min() const;

  /**
   * @return the maximum energy allowed for this layer.
   */
  double energy_max() const;

  /**
   * @return the layer thickness.
   */
  double thickness() const;

  /**
   * @return the layer dimensionless thickness.
   */
  double zcoat() const;

  /**
   * @return the relative bulk density for this layer.
   *     1.0 is nominal full bulk density
   */
  double bulk_density_factor() const;

  /**
   * Dumps layer information to a stream.
   *
   * @param os stream.
   * @param pre optional prefix string.
   * @param pst optional postfix string.
   */
  std::ostream& dump_on( std::ostream& os, char const pre[] = "", 
                                           char const pst[] = "" ) const;

  /**
   * Dumps layer information to a C-style FILE* stream.
   *
   * @param of output file.
   * @param pre optional prefix string.
   * @param pst optional postfix string.
   */
  void cdump_on( std::FILE* of, char const pre[] = "", char const pst[] = "" ) const;

  /**
   * Dumps layer information and constraints to a C-style FILE* stream.
   *
   * @param of output file.
   * @param pre optional prefix string.
   * @param pst optional postfix string.
   */
  void cprint_constraints_on( std::FILE* of, 
                              char const pre[] = "", 
                              char const pst[] = "" ) const;


private:

  /**
   * Evaluate Debye-Waller roughness factor (SAO variant of Windt's DW, 
   * taking only the real part of the factor)
   *
   * @param upper  adjacent layer above this layer, where above
   *            means closer to the vacuum.
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *            and the top surface of the multilayer.
   */
  void
  apply_DebyeWaller_RSAO_factor( 
                  rl_DielectricLayer const& upper,  // upper layer
                  double                    sinphi  // sin(graze angle)
                               );

  /**
   * Evaluate Debye-Waller roughness factor (SAO variant of Windt's DW, 
   * using the complex factor)
   *
   * @param upper  adjacent layer above this layer, where above
   *        means closer to the vacuum.
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *        and the top surface of the multilayer.
   */
  void
  apply_DebyeWaller_CSAO_factor( 
                  rl_DielectricLayer const& upper,  // upper layer
                  double                    sinphi  // sin(graze angle)
                               );

  /**
   * Evaluate Debye-Waller roughness factor (Variant on the DW arising  
   * from a paper by Spiller.)
   *
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *        and the top surface of the multilayer.
   */
  void
  apply_DebyeWaller_Spiller_factor( double sinphi  // sin(graze angle)
                                  );
  /**
   * Evaluate the ``modified Debye-Waller'' roughness factor;
   * phase_factor is \f$2\pi\f$ srough / lambda.
   *
   * @param upper  adjacent layer above this layer, where above
   *        means closer to the vacuum.
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *        and the top surface of the multilayer.
   */
  void
  apply_ModifiedDW_factor( rl_DielectricLayer const& upper,  // upper layer
                           double                    sinphi  // sin(graze angle)
                         );

  /**
   * Evaluate the Nevot-Croce factor.
   *
   * \f[ 
   *    \mathit{NC}_{j-1,j}
   *          =  \exp\left[ -2 {\bf k}_0^2
   *                \left[ \left( \sin^2 \tilde\alpha_0 + \Delta\epsilon_{j}   
   *                       \right)
   *                       \left( \sin^2 \tilde\alpha_0 + \Delta\epsilon_{j-1} 
   *                       \right)
   *                \right]^{1\over 2}
   *                     \sigma_{j}^2 
   *                \right] 
   * \f]
   *
   * phase_factor is \f$2\pi\f$ srough / lambda.
   *
   * @param upper  adjacent layer above this layer, where above
   *            means closer to the vacuum.
   * @param sinphi sine of the graze angle between the ray (in vacuum)
   *            and the top surface of the multilayer.
   */
  void
  apply_NevotCroce_factor( rl_DielectricLayer const& upper,  // upper layer
                           double                    sinphi  // sin(graze angle)
                         );
};

inline rl_Traits::complex const& rl_DielectricLayer::
propagator() const
{ return prop_; }

inline double rl_DielectricLayer::
alpha() const
{ return alpha_; }

inline double rl_DielectricLayer::
gamma() const
{ return gamma_; }

inline double rl_DielectricLayer::
thickness() const
{ return thickness_; }

inline double rl_DielectricLayer::
zcoat() const
{ return zcoat_; }

inline rl_Traits::ERoughType rl_DielectricLayer::
roughness_type() const
{ return rtype_; }

inline double rl_DielectricLayer::
roughness() const
{ return srough_; }

inline rl_ReflectionCoefPOD const& rl_DielectricLayer::
reflection_coef() const
{ return rflcoef_; }

inline double rl_DielectricLayer::
reflectivity( double polarization_factor ) const
{ return rflcoef_.reflectivity( polarization_factor ); }

inline rl_Traits::Bool rl_DielectricLayer::
is_substrate() const
{ return is_substrate_; }

inline rl_Traits::Bool rl_DielectricLayer::
is_vacuum() const
{ return diel_info_.is_vacuum(); }

inline char const* rl_DielectricLayer::
layer_name() const
{ if ( name_ ) { return name_; } else { return ""; } }

inline double rl_DielectricLayer::
energy_min() const
{ return diel_info_.energy_min(); }

inline double rl_DielectricLayer::
energy_max() const
{ return diel_info_.energy_max(); }

inline double rl_DielectricLayer::
bulk_density_factor() const
{ return diel_info_.bulk_density_factor(); }

// rl_DielectricLayer_h_INCLUDED
#endif
