#ifndef rl_Ray_h_INCLUDED
#define rl_Ray_h_INCLUDED

// File:   rl_Ray.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <rl_basicray/rl_BasicRay.h>                  // rl_BasicRay
#include <rl_raylib/rl_Polarization.h>                // rl_Polarization
#include <dvm3/dvm3_vector.h>                         // dmv3_Vector

//########################################################################
// rl_Ray
//########################################################################

/** 
 * \class rl_Ray
 *
 * An rl_BasicRay with added polarization information.
 */

class rl_Ray
  : public rl_BasicRay
{
private:

  /// the polarization state
  rl_Polarization  pol_;

public:


  /**
   * Destructor
   */
 virtual ~rl_Ray();

  /**
   * Default constructor.  Constructs a ray in an INVALID state;
   * use init_ray to initialize the fields.
   */
 rl_Ray();

  /**
   * Constructor.  Construct a ray given an rl_BasicRay and a 
   * polarization state.
   */
  rl_Ray( rl_BasicRay const& ray, rl_PolCSPOD const& cspol );

  /**
   * Constructor.  Construct a ray.
   *
   * @param pos    ray position vector.
   * @param dir    ray direction unit vector.
   * @param energy ray energy (keV).
   * @param id     a numeric ray identifier.
   * @param cspol  an OSAC-style complex polarization amplitude vector.
   */
  rl_Ray( dvm3_Vector const& pos, dvm3_Vector const& dir,
          double energy, long int id,
          rl_PolCSPOD const& cspol );

  /**
   * Initialize a ray.
   *
   * @param pos    ray position vector.
   * @param dir    ray direction unit vector.
   * @param energy ray energy (keV).
   * @param id     a numeric ray identifier.
   * @param cspol  an OSAC-style complex polarization amplitude vector.
   */
  void init_ray( dvm3_Vector const& pos, dvm3_Vector const& dir,
                 double energy, long int id,
                 rl_PolCSPOD const& cspol );

  /**
   * Return the polarization state.
   *
   * @return polarization state
   */
  rl_Polarization const& polarization() const;

  /**
   * Set the polarization state.
   *
   * @param cspol an OSAC-style complex polarization amplitude vector.
   */
  void set_polarization( rl_PolCSPOD const& cspol );

  /**
   * Evaluate rl_PolCSPOD corresponding to this ray's polarization state.
   *
   * @param cs return an OSAC-style complex polarization amplitude vector.
   */
  void get_PolCSPOD( rl_PolCSPOD& cs ) const;

  /**
   * Returns this ray's normalized intensity (i.e., "weight")
   * 
   * @return this ray's normalized intensity (i.e., "weight")
   */
  double intensity() const;

  /**
   * Attenuate this ray by by_how_much
   * 
   * @param by_how_much how much to attenuate this ray.
   */
  void attenuate( double by_how_much );

  /**
   * Reflect this ray's direction vector and polarization state.
   * 
   * @param normal  surface normal unit vector.
   * @param rflcoef complex reflectances for the surface.
   */
  void reflect( dvm3_Vector const& normal, 
                rl_ReflectionCoefPOD const& rflcoef );

  /**
   * Translate to BCS origin and rotate from STD to BCS coordinates.
   *
   * @param trans translation vector
   * @param rotmat rotation matrix to be applied.  rotmat specifies
   *     a rotation from std to bcs.
   */ 
  void translate_rotate( dvm3_Vector const& trans,
                         dvm3_RotMat const& rotmat );

  /**
   * Derotate back to std coordinates; detranslate back to std origin
   * 
   * @param trans  translation vector.
   * @param rotmat rotation matrix to be applied.  rotmat specifies
   *     a rotation from std to bcs; the inverse of rotmat is
   *     applied to the ray.
   */
  void derotate_detranslate( dvm3_Vector const& trans,
                             dvm3_RotMat const& rotmat );

  /**
   * Write the ray contents with optional pre and post comment strings.
   *
   * @param os output stream.
   * @param pre optional prefix c-string.
   * @param pst optional postfix c-string.
   */
  std::ostream& print_on( std::ostream& os, char const pre[] = "", 
                                            char const pst[] = "" ) const;
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//=========================================================================
// dtor, ctors, initializers...

//-------------------------------------------------------------------------
inline rl_Ray::
rl_Ray()
{}

//-------------------------------------------------------------------------
inline rl_Ray::
rl_Ray( rl_BasicRay const& ray, rl_PolCSPOD const& cspol )
  : rl_BasicRay( ray ), pol_( cspol, ray.direction() )
{}

inline rl_Ray::
rl_Ray( dvm3_Vector const& pos, dvm3_Vector const& dir,
        double energy, long int id,
        rl_PolCSPOD const& cspol )
  : rl_BasicRay( pos, dir, energy, id ),
    pol_( cspol, dir )
{}

inline void rl_Ray::
init_ray( dvm3_Vector const& pos, dvm3_Vector const& dir,
          double energy, long int id,
          rl_PolCSPOD const& cspol )
{ 
  rl_BasicRay::init( pos, dir, energy, id ); 
  pol_.init( cspol, dir );
}

//=========================================================================
// accessors

//-------------------------------------------------------------------------
inline rl_Polarization const& rl_Ray::
polarization() const
{ return pol_; }

//-------------------------------------------------------------------------
inline void rl_Ray::
get_PolCSPOD( rl_PolCSPOD& cs ) const
{ pol_.get_PolCSPOD( cs, dir_); }

//-------------------------------------------------------------------------
inline double rl_Ray::
intensity() const
{ return pol_.intensity(); }

//=========================================================================
// mutators

//-------------------------------------------------------------------------
inline void rl_Ray::
set_polarization( rl_PolCSPOD const& cspol )
{ pol_.init( cspol, dir_ ); }

//-------------------------------------------------------------------------
inline void rl_Ray::
attenuate( double by_how_much )
{ pol_.attenuate( by_how_much ); }

// rl_Ray_h_INCLUDED
#endif
