#ifndef rl_Multilayer_rdb_h_INCLUDED
#define rl_Multilayer_rdb_h_INCLUDED

// File:   rl_Multilayer_rdb.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_ray
 *
 * rl_ray is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_ray is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <cstddef>                        // ptrdiff_t
#include <rl_raylib/rl_Traits.h>          // rl_Traits::rl_DielectricPOD
#include <rl_raylib/rl_Multilayer.h>      // rl_Multilayer

//########################################################################
// rl_Multilayer_rdb
//########################################################################


//
 /***
 * /class rl_Multilayer_rdb
 *
 * Construct or initialize rl_Multilayer based on contents of an /rdb table.
 * The rdb table is assumed to contain one row for each layer
 * of the multilayer, ordered from the top (adjacent to vacuum) layer
 * to the substrate (conventionally with thickness of 1.e30).
 * The rdb table is assumed to contain the following columns:
 * - material: an ascii string containing the compostion of
 *   the layer; used for labeling and i/o.  This column is required.
 * - thickness: thickness of the layer in Angstroms
 * - bulkdensity: relative bulk density of the layer; 1.0 is full
 *   nominal bulk density.  This column is required.
 * - roughness_type: type of interlayer grading. 
 *   This column is optional; if it is not present, "none" is assumed.
 *   Currently acceptable values are (case-insensitive):
 *   - none: no interlayer grading
 *   - DW_RSAO: Debye-Waller factor (SAO variant on Windt's DW,
 *              taking only the real part of the factor)
 *   - DW_CSAO: Debye-Waller factor (SAO variant on Windt's DW,
 *              using the complex factor)
 *   - DW_SPILLER: Debye-Waller factor (variant on DW from a paper by
 *                 Spiller (1988)
 *   - MDW: "Modified Debye-Waller" factor
 *   - NC: Nevot-Croce factor
 * - srough: interlayer grading "roughness" sigma (in Angstroms);
 *    This column is required only if the roughness_type column is present.
 * - pathname</code> - This contains a string containing the filename
 *       (optionally prefixed with a path) for the /rdb table containing the
 *       optical constant data for this layer.  The layer is optical 
 *       constants are initialized using rl_DielectricPOD_rdb to read 
 *       the /rdb data.
 */
class rl_Multilayer_rdb
  : public rl_Multilayer
{
public:

  /**
   * Destructor.
   */
 ~rl_Multilayer_rdb();

  /**
   * Constructor.
   *
   * @param rdb_file name of the /rdb file to be read.
   * @param interp_mode interpolation type for the optical constants.
   *
   * If rdb_file is a nonempty string, read in energy, alpha, and gamma
   * from the specified /rdb table.  The array is sorted on the
   * energy field.  If rdb_file is an empty string, init_from_rdb must 
   * be called to initialize the object.
   *
   * The interp_mode argument specifies the form of energy/optical constant
   * interpolation (see rl_Traits in rl_raylib).
   */
  rl_Multilayer_rdb( char const rdb_file[], rl_Traits::EInterpMode interp_mode );

  /**
   * Initializer.  
   *
   * @param rdb_file name of the /rdb file to be read.
   * @param interp_mode interpolation type for the optical constants.
   *
   * rdb_file</code> specifies the name of an /rdb table;
   * energy, alpha, and gamma are read in from the specified /rdb table.  
   * The array is sorted on the energy field.
   *
   * The interp_mode argument specifies the form of energy/optical constant
   * interpolation (see rl_Traits in rl_raylib).
   */
  void init_from_rdb( char const rdb_file[], rl_Traits::EInterpMode interp_mode );
};

// rl_Multilayer_rdb_h_INCLUDED
#endif
