#ifndef vm_vmath_h_INCLUDED
#define vm_vmath_h_INCLUDED

// File:   vm_vmath.h
// Author: Terry Gaet

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of vm_math.cvs
 *
 * vm_math.cvs is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * vm_math.cvs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * Description: collection of methods for vm_VMath
 *
 * History
 *--------
 * 0.0.0 1998-Jan-30  tjg  original version
 */

#include <cstring>       // memcpy strlen
#include <cmath>         // sqrt fabs
#include <iostream>      // ostream
#include <cstdio>        // FILE*

//########################################################################
// vm_VMath<T_fp,N_len>
//########################################################################

 //class vm_VMath<T_fp,N_len> vm_math.h <vm_math/vm_vmath.h>
/** 
 * \class vm_VMath vm_math.h <vm_math/vm_vmath.h>
 *
 * A template class providing common numerical operations on N_len-long 
 * 1 dimensional arrays of T_fp .
 * 
 * T_fp  is a floating point type.
 *
 * N_len is the length of the vector.
 *
 * The array data are assumed to be stored as a contiguous one-dimensional
 * array of N_len T_fp's, properly aligned for type T_fp.
 *
 * Unless otherwise noted, the operations are component by component, 
 * e.g.,
   \verbatim
       vm_VMath<float,4>::mul(prod, v1, v2)
   \endverbatim
 * corresponds to 
   \verbatim
       prod[i] = v1[i] * v2[i], where i = 0,1,2,3.
   \endverbatim
 *
 *  vm_VMath has only static member functions; there are no data members.
 *
 *  Where possible, the static member functions are inlined.
 */

template <class T_fp, int N_len>
class vm_VMath
{
private:

  enum ENelts_     { ENelts     = N_len };

public:

  /**
   * \typedef value_type
   *
   * a typedef for the floating point type;
   */
  typedef T_fp value_type;

  /*!
   * \defgroup ndvec_copy Copy routines
   */
  /*@{*/

  /**
   * Normalize a vector v.
   *
   * @param v      destination vector
   * @param cv     source vector
   *
   * REQUIREMENT:  *v and *cv each has a length of at least N_len
   * contiguous T_fps and is appropriately aligned for T_fps.
   */
  inline static void copy( T_fp v[], T_fp const cv[] );
  /*@}*/

  /*!
   * \defgroup ndvec_setfuncs Set functions
   */
  /*@{*/
  /**
   * Set all elements of v to r.
   *
   * @param v      vector to be set
   * @param r      value
   */
  inline static void set( T_fp v[], T_fp r );
  /*@}*/

  /*!
   * \defgroup ndvec_opeq Componentwise op_eq operations (+=, -=, *=, /=)
   */
  /*@{*/
  /**
   * component-wise v[n] += cv[n]
   *
   * @param v      LHS vector
   * @param cv     RHS vector
   */
  inline static void add_eq( T_fp v[], T_fp const cv[] );

  /**
   * component-wise v[n] -= cv[n]
   *
   * @param v      LHS vector
   * @param cv     RHS vector
   */
  inline static void sub_eq( T_fp v[], T_fp const cv[] );

  /**
   * component-wise v[n] *= cv[n]
   *
   * @param v      LHS vector
   * @param cv     RHS vector
   */
  inline static void mul_eq( T_fp v[], T_fp const cv[] );

  /**
   * component-wise v[n] /= cv[n]
   *
   * @param v      LHS vector
   * @param cv     RHS vector
   */
  inline static void div_eq( T_fp v[], T_fp const cv[] );

  /**
   * component-wise v[n] += r
   *
   * @param v      LHS vector
   * @param r      RHS T_fp
   */
  inline static void add_eq( T_fp v[], T_fp r );

  /**
   * component-wise v[n] -= r
   *
   * @param v      LHS vector
   * @param r      RHS T_fp
   */
  inline static void sub_eq( T_fp v[], T_fp r );

  /**
   * component-wise v[n] *= r
   *
   * @param v      LHS vector
   * @param r      RHS T_fp
   */
  inline static void mul_eq( T_fp v[], T_fp r );

  /**
   * component-wise v[n] /= r
   *
   * @param v      LHS vector
   * @param r      RHS T_fp
   */
  inline static void div_eq( T_fp v[], T_fp r );

  /**
   * component-wise negation of v
   *
   * @param v      vector
   */
  inline static void negate( T_fp v[] );
  /*@}*/

  /*!
   * \defgroup ndvec_mathop {add,sub,mul,div}:  componentwise binary operations.
   */
  /*@{*/
  /**
   * component-wise v[n] = cv1[n] + cv2[n]
   *
   * @param v      result vector
   * @param cv1    1st input vector
   * @param cv2    2nd input vector
   */
  inline static void add( T_fp v[], T_fp const cv1[], T_fp const cv2[] );

  /**
   * component-wise v[n] = cv1[n] - cv2[n]
   *
   * @param v      result vector
   * @param cv1    1st input vector
   * @param cv2    2nd input vector
   */
  inline static void sub( T_fp v[], T_fp const cv1[], T_fp const cv2[] );

  /**
   * component-wise v[n] = cv1[n] * cv2[n]
   *
   * @param v      result vector
   * @param cv1    1st input vector
   * @param cv2    2nd input vector
   */
  inline static void mul( T_fp v[], T_fp const cv1[], T_fp const cv2[] );

  /**
   * component-wise v[n] = cv1[n] / cv2[n]
   *
   * @param v      result vector
   * @param cv1    1st input vector
   * @param cv2    2nd input vector
   */
  inline static void div( T_fp v[], T_fp const cv1[], T_fp const cv2[] );

  /**
   * component-wise v[n] = cv[n] + r
   *
   * @param v      result vector
   * @param cv     input vector
   * @param r      input T_fp
   */
  inline static void add( T_fp v[], T_fp const cv[], T_fp r );

  /**
   * component-wise v[n] = cv[n] - r
   *
   * @param v      result vector
   * @param cv     input vector
   * @param r      input T_fp
   */
  inline static void sub( T_fp v[], T_fp const cv[], T_fp r );

  /**
   * component-wise v[n] = cv[n] * r
   *
   * @param v      result vector
   * @param cv     input vector
   * @param r      input T_fp
   */
  inline static void mul( T_fp v[], T_fp const cv[], T_fp r );

  /**
   * component-wise v[n] = cv[n] / r
   *
   * @param v      result vector
   * @param cv     input vector
   * @param r      input T_fp
   */
  inline static void div( T_fp v[], T_fp const cv[], T_fp r );

  /**
   * component-wise v[n] = r + cv[n]
   *
   * @param v      result vector
   * @param r      input T_fp
   * @param cv     input vector
   */
  inline static void add( T_fp v[], T_fp r, T_fp const cv[] );

  /**
   * component-wise v[n] = r - cv[n]
   *
   * @param v      result vector
   * @param r      input T_fp
   * @param cv     input vector
   */
  inline static void sub( T_fp v[], T_fp r, T_fp const cv[] );

  /**
   * component-wise v[n] = r * cv[n]
   *
   * @param v      result vector
   * @param r      input T_fp
   * @param cv     input vector
   */
  inline static void mul( T_fp v[], T_fp r, T_fp const cv[] );

  /**
   * component-wise v[n] = r / cv[n]
   *
   * @param v      result vector
   * @param r      input T_fp
   * @param cv     input vector
   */
  inline static void div( T_fp v[], T_fp r, T_fp const cv[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup ndvec_lincomb Linear combinations
   */
  /*@{*/
  /**
   * form linear combination:  res = c1 * v1 + c2 * v2.
     \verbatim
       For each i:  res[i] = c1 * v1[i] + c2 * v2[i]
     \endverbatim
   *
   * @param res    result vector
   * @param c1     1st T_fp
   * @param v1     1st vector
   * @param c2     2nd T_fp
   * @param v2     2nd vector
   */
  inline static void lincomb( T_fp res[],
                              T_fp c1, T_fp const v1[],
                              T_fp c2, T_fp const v2[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup ndvec_io I/O operations.
   */
  /*@{*/
  /**
   * Print a vector to an ostream.
   *
   * @param os       the ostream
   * @param v        vector to be printed
   * @param by       stride by this many
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  inline static std::ostream&
  print_on( std::ostream& os, T_fp const v[], int by,
            char const prefix[] = "", char const postfix[] = "" );

  /**
   * Print a vector to a FILE* stream.
   *
   * @param of       the FILE*
   * @param v        vector to be printed
   * @param by       stride by this many
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  inline static void
  cprint_on( FILE* of, T_fp const v[], int by,
             char const prefix[] = "", char const postfix[] = "" );
  /*@}*/
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

//-------------------------------------------------------------------------
// copy routines

template <class T_fp, int N_len>
inline void vm_VMath<T_fp,N_len>::
copy( T_fp v[], T_fp const cv[] )
{ memcpy( v, cv, N_len * sizeof(T_fp) ); }

//-------------------------------------------------------------------------
// set functions

template <class T_fp, int N_len>
inline void vm_VMath<T_fp,N_len>::
set( T_fp v[], T_fp r )
{ int n = N_len+1; while ( --n ) { *v++ = r; } }

//-------------------------------------------------------------------------
// {add,sub,mul,div}_eq:  componentwise operations.

#define _tpl_Op_EQ_(_OPNAM_,_OPEQ_) \
  template <class T_fp, int N_len> \
  inline void vm_VMath<T_fp,N_len>::_OPNAM_( T_fp v[], T_fp const cv[] ) \
  { int n = N_len+1; while ( --n ) { *v++ _OPEQ_ *cv++; } } \
  template <class T_fp, int N_len> \
  inline void vm_VMath<T_fp,N_len>::_OPNAM_( T_fp v[], T_fp r ) \
  { int n = N_len+1; while ( --n ) { *v++ _OPEQ_ r; } }
  _tpl_Op_EQ_(add_eq,+=)
  _tpl_Op_EQ_(sub_eq,-=)
  _tpl_Op_EQ_(mul_eq,*=)
  _tpl_Op_EQ_(div_eq,/=)
#undef _tpl_Op_EQ_

template <class T_fp, int N_len>
inline void vm_VMath<T_fp,N_len>::
negate( T_fp v[] )
{ int n = N_len+1; while ( --n ) { *v = - *v; ++v; } }

//-------------------------------------------------------------------------
// {add,sub,mul,div}:  componentwise binary operations.
#define _tpl_BinOp_(_OPNAM_,_OP_) \
  template <class T_fp, int N_len> \
  inline void vm_VMath<T_fp,N_len>::_OPNAM_( T_fp v[], \
			 	             T_fp const cv1[], T_fp const cv2[] ) \
  { int n = N_len+1; while ( --n ) { *v++ = *cv1++ _OP_ *cv2++; } } \
  template <class T_fp, int N_len> \
  inline void vm_VMath<T_fp,N_len>::_OPNAM_( T_fp v[], T_fp const cv[], T_fp r ) \
  { int n = N_len+1; while ( --n ) { *v++ = *cv++ _OP_ r; } } \
  template <class T_fp, int N_len> \
  inline void vm_VMath<T_fp,N_len>::_OPNAM_( T_fp v[], T_fp r, T_fp const cv[] ) \
  { int n = N_len+1; while ( --n ) { *v++ = r _OP_ *cv++; } }
  _tpl_BinOp_(add,+)
  _tpl_BinOp_(sub,-)
  _tpl_BinOp_(mul,*)
  _tpl_BinOp_(div,/)
#undef _tpl_BinOp_

//-------------------------------------------------------------------------
// linear combinations

template <class T_fp, int N_len>
inline void vm_VMath<T_fp,N_len>::
lincomb( T_fp res[],
         T_fp c1, T_fp const v1[],
         T_fp c2, T_fp const v2[] )
{ int n = N_len+1; while ( --n ) { *res++ = c1 * *v1++ + c2 * *v2++; } }

//-------------------------------------------------------------------------
// i/o

template <class T_fp, int N_len>
inline std::ostream& vm_VMath<T_fp,N_len>::
print_on( std::ostream& os, T_fp const v[], int by,
          char const prefix[], char const postfix[] )
{
  if ( std::strlen(prefix) ) { os << prefix; }
  int i = N_len / by;
  while ( i-- )
  {
    int j = by;
    while ( --j )
    {
      os << *v++ << " ";
    }
    os << *v++ << "\n";
  }
  if ( std::strlen(postfix) ) { os << postfix; }
  return os;
}

template <class T_fp, int N_len>
inline void vm_VMath<T_fp,N_len>::
cprint_on( FILE* of, T_fp const v[], int by,
          char const prefix[], char const postfix[] )
{
  if ( std::strlen(prefix) ) { std::fprintf(of, "%s", prefix); }
  int i = N_len / by;
  while ( i-- )
  {
    int j = by;
    while ( --j )
    {
      fprintf(of, "%.18e ", *v++);
    }
    fprintf(of, "%.18e\n", *v++);
  }
  if ( std::strlen(postfix) ) { std::fprintf(of, "%s", postfix); }
}

#endif  /* vm_vmath.h */
