// -*-c++-*-

// --8<--8<--8<--8<--
//
// Copyright (C) 2013 Smithsonian Astrophysical Observatory
//
// This file is part of bpipexx
//
// bpipexx is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or (at
// your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -->8-->8-->8-->8--

#ifndef BPIPEXX_H
#define BPIPEXX_H

#include <iostream>
#include <sstream>
#include <string>
#include <vector>

#include <bpipe/bpipe.h>
#include <bpipe/datatypes.h>

#include <Exception/Exception.h>

using namespace std;

namespace SAOTrace {

    namespace BPipeX {

	/**
	   \struct SAOTrace::BPipeX::HdrF
	   \headerfile bpipexx/bpipexx.h

	   A simple structure to hold information about a \c bpipe header field.
	   Note that the existing constructors set the \c matrix field to \c NULL.

	   \tparam T The storage type of the data. This is one of the
	   intrinsic types and structures supported by the \c bpipe library.

	**/
	template< typename T >
	struct HdrF {

	    string name;
	    BPMatrix *matrix;
	    BPDataType BPT;
	    T datum;
	    bool copy;

	    /** \brief Construct a \e scalar field with the given name, data, and type.

		\tparam T The storage type of the data. This is one of the
		intrinsic types and structures supported by the \c bpipe library.
		\param[in] cname  The name of the field.
		\param[in] idatum  The data to store.  This is copied.
	    **/
	    HdrF(const char* cname, T idatum );

	    /** \brief Return a pointer to the data

		\returns a pointer to the copy of the data
	    **/
	    T* data() {  return &datum; }

	};



	/**
	   \struct SAOTrace::BPipeX::DpktF
	   \headerfile bpipexx/bpipexx.h

	   A simple structure to hold information about a \bpipe data field.
	   Note that the existing constructors set the \c matrix field to \c NULL.

	   \tparam T The storage type of the data. This is one of the
	   intrinsic types and structures supported by the \c bpipe library.

	**/
	template< typename T >
	struct DpktF {

	    string name;
	    BPMatrix *matrix;
	    BPDataType BPT;

	    /** \brief Construct a \e scalar field with the given name and type

		\tparam T The storage type of the data. This is one of the
		intrinsic types and structures supported by the \c bpipe library.
	       \param[in] cname  The name of the field.
	    **/
	    DpktF( const char* cname );

	};


	class Simple {

	public:

	    ~Simple( );
	    Simple( );

	    Simple( const string& input, const string& output );

	    /** \brief add a header field

		For example:
		\code
		bpipe.add( HdrF<double>("temperature", 33.2) );
		\endcode

		\param[in] field A \c BPipeX::HdrF object

		\throw Exception if unable to create the header field

	    **/
	    template<typename T>
	    void add( HdrF<T> field) {

		if ( bpipe_hdrf_add( _bpipe, field.name.c_str(),
				     field.BPT, field.matrix,
				     field.data(), field.copy ) )
		    throw Exception( bpipe_strerror( bpipe_errno ) );
	    }

	    /** \brief add a data packet field

		For example:
	       \code
	       bpipe.add( DpktF<double>("smile") );
	       \endcode

	       \param[in] field  A \c BPipeX::DpktF object

		\throw Exception if unable to create the data packet field or the
		                 field exists with a different data type.
	    **/
	    template<typename T>
	    void add( const DpktF<T>& field ) {

		int res = bpipe_dpktf_add( _bpipe, field.name.c_str(),
					   field.BPT, field.matrix );
		if ( res < 0 )
		    throw Exception( bpipe_strerror( bpipe_errno ) );

		if ( res == 1 ) {

		    DpktField *dpktf = bpipe_dpktf( _bpipe, field.name.c_str() );

		    if ( NULL == dpktf )
			throw Exception( string("internal error: " ) + field.name + "should exist but doesn't" );

		    if ( bpipe_dpktf_type( dpktf) != field.BPT ) {
			ostringstream errstr;
			errstr << "requested type of " << field.name << "(" << field.BPT << ")"
			       << "not equal to type of existing field" << "(" << bpipe_dpktf_type( dpktf ) << ")";

			throw Exception( errstr.str() );
		    }

		}


	    }

	    void delete_hdrf( const char* name, size_t index=BPHdrfIdx_ALL );
	    void delete_dpktf( const char* name,
			       BPDataSite site = BPDSite_OUTPUT,
			       BPipeOutput *channel = BPOutputChannel_ALL );


	    bool has_hdrf( const char* name, size_t index=BPHdrfIdx_LAST );
	    bool has_dpktf( const char* name );



	    /** \brief get a pointer to the data in a data packet field

		For example:
	       \code
	       double* temp_p = bpipe.dpktf_data<double>( "temp" );
	       \endcode
	       \tparam T the type of the stored data.
	       \param[in] name  the name of the data packet field.
	    **/
	    template<typename T>
	    T* dpktf_data( const char* name ) {

		DpktField *dpktf = bpipe_dpktf( _bpipe, name );

		if ( NULL == dpktf )
		    throw Exception( string("non-existent data packet field: ") + name );

		return static_cast<T*>( static_cast<void*>( bpipe_dpktf_data( dpktf, core_image ) ) );
	    }

	    /** \brief get a pointer to the data in a header packet field

		For example:
	       \code
	       double* temp_p = bpipe.hdrf_data<double>( "temp" );
	       \endcode
	       \tparam T the type of the stored data.
	       \param[in] name  the name of the data packet field.
	       \param[in] idx   the index of the field. Defaults to the last one.
	    **/
	    template<typename T>
	    T* hdrf_data( const char* name, size_t idx = BPHdrfIdx_LAST ) {

		void *data = bpipe_hdrf_data( _bpipe, name, idx );

		if ( NULL == data )
		    throw Exception( string("non-existent data packet field: ") + name );

		return static_cast<T*>( data );
	    }
	    void map( int npkts = 1 );

	    size_t read_dpkts ();

	    void write_hdr ();

	    void write_dpkts( BPipeOutput* bpo = BPOutputChannel_ALL );

	    BPipe* bpipe( ) { return _bpipe ; }


	private:
	    std::vector< BPipeOutput* > bpo;

	    BPipe* _bpipe;
	    void* core_image;
	    size_t core_image_size;
	    int n_pkts;

	};

    }
}



#endif // ! BPIPEXX_H
