#ifndef dvm3_matrix_h_INCLUDED
#define dvm3_matrix_h_INCLUDED

// File:    dvm3_matrix.h
// Author:  Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of dvm3
 *
 * dvm3 is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * dvm3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */


/****************************************************************************
 * Description: methods for 3-matrix routines
 */

#include <dvm3/dvm3_vector.h>           // dvm3_Vector
#include <vm_math/vm_m3math.h>          // vm_M3Math<T>
#include <iosfwd>
#include <cstdio>

//########################################################################
// dvm3_MPOD
//########################################################################

/*@{*/
/**
 * \internal
 *
 * \struct dvm3_MPOD dvm3_matrix.h <dvm3/dvm3_matrix.h>
 *
 * dvm3_MPOD is a Plain Ol' Data base class to represent the matrix data.
 * dvm3_MPOD is declared as a struct encapsulating an array of doubles
 * in order to assure that the data alignment and contiguity are maintained.
 * This implementation is exposed purely for efficiency considerations; users
 * shall not assume that this implementation detail will not change in future.
 * DO NOT rely on the fact that there is an array under the hood or the
 * naming of the data members within dvm3_MPOD.
 *
 * NOTE:  a POD class must satisfy (Jack W. Reeves, C++Report, February 1998,
 *         p. 46-54):
 *  - no user-declared constructors, copy-assign operator, or destructor
 *  - no private or protected nonstatic data members
 *  - no base classes
 *  - no virtual functions
 *  - no nonstatic data members of type pointer to member,
 *      non-POD-struct/class (or array of such types) or reference
 */
struct dvm3_MPOD
{
  double m_[9];
};
/*@}*/

//########################################################################
// dvm3_Matrix
//########################################################################

/** 
 * \class dvm3_Matrix dvm3_matrix.h <dvm3/dvm3_matrix.h>
 * 
 * A class representing 3x3 matrix of doubles with common numerical 
 * operations * defined.  The dvm3_Matrix class is a simple class 
 * to handle common * numerical operations on 3-matrices of doubles.  
 * Unless otherwise noted, * the operations are component by 
 * component operations, * e.g., 
   \verbatim
   v1 * v2 = { v1[0] * v2[0], v1[1] * v2[1], v1[2] * v2[2] }
   \endverbatim
 *
 * This is intended as a small component which can be inexpensively created
 * on the stack; the constructors and destructor do not make use of
 * dynamic allocation.  The dvm3_Matrix default constructor does not 
 * initialize the memory.
 *
 * dvm3_Matrix works on any of these representations of 3-vectors:
 * - dvm3_Vector: a class encapsulating a 3-vector of doubles with
 *                   associated mathematical operations
 * - double*:     a c-style array holding (at least) 3 doubles.
 *
 * NOTE:  although dvm3_Matrix works with either of the above two vector 
 * types, the SAME vector type must be used for ALL occurrences within
 * any given method call.
 */
class dvm3_Matrix
{
protected:

  dvm3_MPOD data_;
  enum      ENelts_     { ENelts = 9 };
  enum      ERowOffset_ { ERow0 = 0, ERow1 = 3, ERow2 = 6 };
  enum      EColStride_ { EColStride = 3 };

public:

  // ---------------------------------------------------------------------
  /*!
   * \defgroup dtor_ctors_matrix Destructor; Constructors
   */
  /*@{*/
  /**
   * Do-nothing destructor.
   */
 ~dvm3_Matrix();

  /**
   * Default constructor; NO INITIALIZATION IS APPLIED.
   */
  dvm3_Matrix();

  /**
   * Copy constructor.
   *
   * @param m     matrix to be copied
   */
  dvm3_Matrix( dvm3_Matrix const& m );

  /**
   * Construct a dvm3_Matrix; initialize by row.
   *
   * @param x     1st row
   * @param y     2nd row
   * @param z     3rd row
   */
  dvm3_Matrix( dvm3_Vector const& x, 
               dvm3_Vector const& y, 
               dvm3_Vector const& z ); 

  /**
   * Construct a dvm3_Matrix; initialize by row.
   *
   * @param x     1st row
   * @param y     2nd row
   * @param z     3rd row
   */
  dvm3_Matrix( double const x[],
               double const y[],
               double const z[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup itors Initializers
   */
  /*@{*/

  /**
   * initialize this dvm3_Matrix by row.
   *
   * @param v1    1st row
   * @param v2    2nd row
   * @param v3    3rd row
   */
  void
  init_by_row( dvm3_Vector const& v1,
               dvm3_Vector const& v2,
               dvm3_Vector const& v3 );

  /**
   * initialize this dvm3_Matrix by row.
   *
   * @param v1    1st row
   * @param v2    2nd row
   * @param v3    3rd row
   */
  void
  init_by_row( double const v1[],
               double const v2[],
               double const v3[] );

  /**
   * initialize this dvm3_Matrix by column.
   *
   * @param v1    1st column
   * @param v2    2nd column
   * @param v3    3rd column
   */
  void
  init_by_col( dvm3_Vector const& v1,
               dvm3_Vector const& v2,
               dvm3_Vector const& v3 );

  /**
   * initialize this dvm3_Matrix by column.
   *
   * @param v1    1st column
   * @param v2    2nd column
   * @param v3    3rd column
   */
  void
  init_by_col( double const v1[],
               double const v2[],
               double const v3[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup ators Accessors
   */
  /*@{*/

  /**
   * Return row i (const version)
   *
   * @return       row i
   * 
   * @param i      index of row to be returned
   */
  double const* operator[](int i) const;
    //
    //: return row i (const version)

  /**
   * Return row i (non-const version)
   *
   * @return       row i
   * 
   * @param i      index of row to be returned
   */
  double* operator[](int i);
    //
    //: return row i (non-const version)

  /**
   * Return element [i][j] (const version)
   *
   * @return       element[i][j]
   * 
   * @param i      index 
   * @param j      index 
   */
  double  operator()(int i, int j) const;

  /**
   * Return element [i][j] (non-const version)
   *
   * @return       element[i][j]
   * 
   * @param i      index 
   * @param j      index 
   */
  double& operator()(int i, int j);
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup assign Assignment, op= operators
   */
  /*@{*/
  /**
   * Copy-assignment operator
   *
   * @return       modified matrix
   *
   * @param rhs    matrix value for assignment
   */
  dvm3_Matrix& operator=( dvm3_Matrix const& rhs );

  /**
   * Assign the value of rhs to each component of this matrix
   *
   * @return       modified matrix
   *
   * @param rhs    double value to be assigned to each component of this matrix
   */
  dvm3_Matrix& operator=( double       rhs );

  /**
   * Component-wise += operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value to be added to each component of this matrix
   */
  dvm3_Matrix& operator+=( dvm3_Matrix const& rhs );

  /**
   * Component-wise -= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value to be subtracted from each component of this matrix
   */
  dvm3_Matrix& operator-=( dvm3_Matrix const& rhs );

  /**
   * Component-wise *= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value by which each component of this matrix is to be
   *               multiplied
   */
  dvm3_Matrix& operator*=( dvm3_Matrix const& rhs );

  /**
   * Component-wise /= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value by which each component of this matrix is to be
   *               divided
   */
  dvm3_Matrix& operator/=( dvm3_Matrix const& rhs );

  /**
   * Component-wise /= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value to be added to each component of this matrix
   */
  dvm3_Matrix& operator+=( double rhs );

  /**
   * Component-wise /= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value to be subtracted from each component of this matrix
   */
  dvm3_Matrix& operator-=( double rhs );

  /**
   * Component-wise *= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value by which each component of this matrix is to be
   *               multiplied
   */
  dvm3_Matrix& operator*=( double rhs );

  /**
   * Component-wise /= operator.
   *
   * @return       modified matrix
   *
   * @param rhs    value by which each component of this matrix is to be
   *               divided
   */
  dvm3_Matrix& operator/=( double rhs );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup unary Unary operators
   */
  /*@{*/
  /**
   * Do-nothing unary +; provided for completeness.
   */
  dvm3_Matrix& operator+();

  /**
   * Unary -; negates each component of this matrix
   */
  dvm3_Matrix& operator-();

  // ---------------------------------------------------------------------
  /*!
   * \defgroup inj_ext Row/Column inject/extract
   */
  /*@{*/
  /**
   * Replace row whichrow by new_row
   *
   * @param whichrow  index of row to be replaced
   * @param new_row   replacement row
   */
  void inject_row( int whichrow, dvm3_Vector const& new_row );

  /**
   * Replace row whichrow by new_row
   *
   * @param whichrow  index of row to be replaced
   * @param new_row   replacement row (C-style vector)
   */
  void inject_row( int whichrow, double const new_row[] );

  /**
   * Replace column whichcol by new_col
   *
   * @param whichcol  index of column to be replaced
   * @param new_col   replacement column 
   */
  void inject_col( int whichcol, dvm3_Vector const& col );

  /**
   * Replace column whichcol by new_col
   *
   * @param whichcol  index of column to be replaced
   * @param new_col   replacement column (C-style vector)
   */
  void inject_col( int whichcol, double const col[] );

  /**
   * Copy row whichrow into supplied dvm3_Vector
   *
   * @param whichrow  index of row to be copied
   * @param row       destination
   */
  void extract_row( int whichrow, dvm3_Vector& row ) const;

  /**
   * Copy row whichrow into supplied vector
   *
   * @param whichrow  index of row to be copied
   * @param row       destination (C-style vector)
   */
  void extract_row( int whichrow, double row[] ) const;

  /**
   * Copy column whichcol into supplied vector
   *
   * @param whichcol  index of column to be copied
   * @param col       destination
   */
  void extract_col( int whichcol, dvm3_Vector& col ) const;

  /**
   * Copy column whichcol into supplied vector
   *
   * @param whichcol  index of column to be copied
   * @param col       destination (C-style vector)
   */
  void extract_col( int whichcol, double* col ) const;

 /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup dyads Vector-vector dyadic product
   */
  /*@{*/
  /**
   * Form dyadic product (outer product) of vectors v1 and v2.
     \verbatim
       For each i, j:  m[i][j] = v1[i] * v2[j]
     \endverbatim
   *
   * @param v1    1st vector
   * @param v2    2nd vector
   */
  void dyad_product( dvm3_Vector const& v1, dvm3_Vector const& v2 );

  /**
   * Form dyadic product (outer product) of vectors v1 and v2.
     \verbatim
       For each i, j:  m[i][j] = v1[i] * v2[j]
     \endverbatim
   *
   * @param v1    1st vector
   * @param v2    2nd vector
   */
  void dyad_product( double const v1[], double const v2[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup lincomb Linear combination
   */
  /*@{*/
  /**
   * Form linear combination:  this = c1 * m1 + c2 * m2.
     \verbatim
       For each i, j:  m[i][j] = c1*m1[i][j] + c2*m2[i][j]
     \endverbatim
   *
   * @param c1    1st scalar
   * @param m1    1st matrix
   * @param c2    2nd scalar
   * @param m2    2nd matrix
   */
  void lincomb( double c1, dvm3_Matrix const& m1,
                double c2, dvm3_Matrix const& m2 );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup matvec Matrix-Vector operations
   */
  /*@{*/
  /**
   * Matrix multiplication of vector v by this matrix.
     \verbatim
       result = *this _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   resultant vector
   * @param v        vector
   */
  void mvmult(  dvm3_Vector& result, dvm3_Vector const& v ) const;

  /**
   * Matrix multiplication of vector v by this matrix.
     \verbatim
       result = *this _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   resultant vector
   * @param v        vector
   */
  void mvmult(  double result[], double const v[] ) const;

  /**
   * Matrix multiplication of vector v by the transpose of this matrix.
     \verbatim
       result = (transpose of *this) _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   resultant vector
   * @param v        vector
   */
  void mtvmult( dvm3_Vector& result, dvm3_Vector const& v ) const;

  /**
   * Matrix multiplication of vector v by the transpose of this matrix.
     \verbatim
       result = (transpose of *this) _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   resultant vector
   * @param v        vector
   */
  void mtvmult( double result[], double const v[] ) const;
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup matmat Matrix-Matrix operations
   */
  /*@{*/
  /**
   * Matrix multiplication of m1 by m2.
     \verbatim
       *this = m1 _matrix_multiply_ m2.
     \endverbatim
   *
   * @param m1    1st matrix
   * @param m2    2nd matrix
   */
  void mmult( dvm3_Matrix const& m1, dvm3_Matrix const& m2 );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \ingroup dyads
   */
  /*@{*/
  /**
   * Form dyadic product (outer product) of vectors v1 and v2.
     \verbatim
       For each i, j:  result[i][j] = v1[i] * v2[j]
     \endverbatim
   *
   * @param result   dyadic product
   * @param v1       1st vector
   * @param v2       2nd vector
   */
  friend void
  dyad_product( dvm3_Matrix& result,
                dvm3_Vector const& v1, dvm3_Vector const& v2 );

  /**
   * Form dyadic product (outer product) of vectors v1 and v2.
     \verbatim
       For each i, j:  result[i][j] = v1[i] * v2[j]
     \endverbatim
   *
   * @param result   dyadic product
   * @param v1       1st vector
   * @param v2       2nd vector
   */
  friend void
  dyad_product( dvm3_Matrix& result, 
                double const v1[], double const v2[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \ingroup lincomb
   */
  /*@{*/
  /**
   * Form linear combination:  result = c1 * m1 + c2 * m2.
     \verbatim
       For each i, j:  result[i][j] = c1*m1[i][j] + c2*m2[i][j]
     \endverbatim
   *
   * @param result  resultant linear combination
   * @param c1      1st scalar
   * @param m1      1st matrix
   * @param c2      2nd scalar
   * @param m2      2nd matrix
   */
  friend void
  lincomb( dvm3_Matrix& result, double c1, dvm3_Matrix const& m1,
                                double c2, dvm3_Matrix const& m2 );

  // ---------------------------------------------------------------------
  /*!
   * \ingroup matvec
   */
  /*@{*/
  /**
   * Matrix multiplication of vector v by matrix m.
     \verbatim
       result = m _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   product
   * @param m        matrix
   * @param v        vector
   */
  friend void
  mvmult( dvm3_Vector& result, dvm3_Matrix const& m, dvm3_Vector const& v );

  /**
   * Matrix multiplication of vector v by matrix m.
     \verbatim
       result = m _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   product
   * @param m        matrix
   * @param v        vector
   */
  friend void
  mvmult( double result[], dvm3_Matrix const& m, double const v[] );

  /**
   * Matrix multiplication of vector v by transpose of matrix m.
     \verbatim
       result = (transpose of m) _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   product
   * @param m        matrix
   * @param v        vector
   */
  friend void
  mtvmult( dvm3_Vector& result, dvm3_Matrix const& m, dvm3_Vector const& v );

  /**
   * Matrix multiplication of vector v by transpose of matrix m.
     \verbatim
       result = (transpose of m) _matrix_multiply_ v.
     \endverbatim
   *
   * @param result   product
   * @param m        matrix
   * @param v        vector
   */
  friend void
  mtvmult( double result[], dvm3_Matrix const& m, double const v[] );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \ingroup matmat
   */
  /*@{*/
  /**
   * Matrix multiplication of m1 by m2.
     \verbatim
    /  result = m1 _matrix_multiply_ m2.
     \endverbatim
   *
   * @param result   product
   * @param m1       1st matrix
   * @param m2       2nd matrix
   */
  friend void
  mmult( dvm3_Matrix& result, dvm3_Matrix const& m1, dvm3_Matrix const& m2 );
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \ingroup assign
   */
  /*@{*/
  /**
   * Component-wise sum of m1 and m2.
   *
   * @return         component-wise sum
   * 
   * @param m1       1st matrix
   * @param m2       2nd matrix
   */
  friend 
  dvm3_Matrix operator+ (dvm3_Matrix const& m1, dvm3_Matrix const& m2);

  /**
   * Component-wise difference of m1 and m2.
   *
   * @return         component-wise difference
   * 
   * @param m1       1st matrix
   * @param m2       2nd matrix
   */
  friend 
  dvm3_Matrix operator- (dvm3_Matrix const& m1, dvm3_Matrix const& m2);

  /**
   * Component-wise product of m1 and m2.
   *
   * @return         component-wise product 
   * 
   * @param m1       1st matrix
   * @param m2       2nd matrix
   */
  friend 
  dvm3_Matrix operator* (dvm3_Matrix const& m1, dvm3_Matrix const& m2);
    //
    //: component-wise product of m1 and m2.

  /**
   * Component-wise division of m1 and m2.
   *
   * @return         component-wise ratio
   * 
   * @param m1       1st matrix
   * @param m2       2nd matrix
   */
  friend 
  dvm3_Matrix operator/ (dvm3_Matrix const& m1, dvm3_Matrix const& m2);
  // ---------------------------------------------------------------------
  /*!
   * \ingroup assign
   */
  /*@{*/
  /**
   * Set each component to d + m[i][j].
   *
   * @return    result
   *
   * @param d   a double
   * @param m   a matrix
   */
  friend dvm3_Matrix operator+ (double d, dvm3_Matrix const& m);

  /**
   * Set each component to d - m[i][j].
   *
   * @return    result
   *
   * @param d   a double
   * @param m   a matrix
   */
  friend dvm3_Matrix operator- (double d, dvm3_Matrix const& m);

  /**
   * Set each component to d * m[i][j].
   *
   * @return    result
   *
   * @param d   a double
   * @param m   a matrix
   */
  friend dvm3_Matrix operator* (double d, dvm3_Matrix const& m);

  /**
   * Set each component to d / m[i][j].
   *
   * @return    result
   *
   * @param d   a double
   * @param m   a matrix
   */
  friend dvm3_Matrix operator/ (double d, dvm3_Matrix const& m);

  /**
   * Set each component to m[i][j] + d.
   *
   * @return    result
   *
   * @param m   a matrix
   * @param d   a double
   */
  friend dvm3_Matrix operator+ (dvm3_Matrix const& m, double d);

  /**
   * Set each component to m[i][j] - d.
   *
   * @return    result
   *
   * @param m   a matrix
   * @param d   a double
   */
  friend dvm3_Matrix operator- (dvm3_Matrix const& m, double d);

  /**
   * Set each component to m[i][j] * d.
   *
   * @return    result
   *
   * @param m   a matrix
   * @param d   a double
   */
  friend dvm3_Matrix operator* (dvm3_Matrix const& m, double d);

  /**
   * Set each component to m[i][j] / d.
   *
   * @return    result
   *
   * @param m   a matrix
   * @param d   a double
   */
  friend dvm3_Matrix operator/ (dvm3_Matrix const& m, double d);
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \ingroup norm
   */
  /*@{*/
  /**
   * Force the matrix to be a proper orthonormal matrix.
   *
   * The y row is replaced by z cross x, and then x is replaced with
   * y cross z.  Normalize all three rows are normalized.
   * This forms a right-handed orthonormal triple.
   *  
   * "Proper", in this context, means that it preserves the parity of the
   * coordinate system, transforming right-handed coordinates into 
   * right-handed coordinates.
   */
  void orthonormalize();
  /*@}*/

  // ---------------------------------------------------------------------
  /*!
   * \defgroup io I/O operations.
   */
  /*@{*/
  /**
   * Print this dvm3_Matrix to an ostream.
   *
   * @param os       the ostream
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  std::ostream& print_on( std::ostream& os, char const pre[]  = "",
                                            char const post[] = "") const;

  /**
   * Formatted dvm3_Matrix output to an ostream.
   *
   * @param os       the ostream
   */
  friend std::ostream& operator<<( std::ostream& os, dvm3_Matrix const& );

  /**
   * Print this dvm3_Matrix to a FILE* stream.
   *
   * @param of       the FILE*
   * @param v        vector to be printed
   * @param prefix   optional prefix string 
   * @param postfix  optional postfix string
   */
  void
  cprint_on( FILE* os, 
             char const prefix[] = "", char const postfix[] = "" );
  /*@}*/
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################

inline dvm3_Matrix::
~dvm3_Matrix()
{}

inline dvm3_Matrix::
dvm3_Matrix()
{}

inline dvm3_Matrix::
dvm3_Matrix( dvm3_Matrix const& other )
{ data_ = other.data_; }

inline dvm3_Matrix::
dvm3_Matrix( dvm3_Vector const& x,
             dvm3_Vector const& y,
             dvm3_Vector const& z )
{
  vm_V3Math<double>::copy( &data_.m_[ERow0], x.data_.x_ );
  vm_V3Math<double>::copy( &data_.m_[ERow1], y.data_.x_ );
  vm_V3Math<double>::copy( &data_.m_[ERow2], z.data_.x_ );
}

inline dvm3_Matrix::
dvm3_Matrix( double const x[],
             double const y[],
             double const z[] )
{
  vm_V3Math<double>::copy( &data_.m_[ERow0], x );
  vm_V3Math<double>::copy( &data_.m_[ERow1], y );
  vm_V3Math<double>::copy( &data_.m_[ERow2], z );
}

inline void dvm3_Matrix::
init_by_row( dvm3_Vector const& x,
             dvm3_Vector const& y,
             dvm3_Vector const& z)
{ vm_M3Math<double>::init_by_row( data_.m_, 
                                   x.data_.x_, y.data_.x_, z.data_.x_ ); }

inline void dvm3_Matrix::
init_by_row( double const x[],
             double const y[],
             double const z[])
{ vm_M3Math<double>::init_by_row( data_.m_, x, y, z ); }

inline void dvm3_Matrix::
init_by_col( dvm3_Vector const& x,
             dvm3_Vector const& y,
             dvm3_Vector const& z)
{ vm_M3Math<double>::init_by_col( data_.m_, 
                                   x.data_.x_, y.data_.x_, z.data_.x_ ); }

inline void dvm3_Matrix::
init_by_col( double const x[],
             double const y[],
             double const z[])
{ vm_M3Math<double>::init_by_col( data_.m_, x, y, z ); }

inline dvm3_Matrix& dvm3_Matrix::
operator=( dvm3_Matrix const& m )
{
  vm_M3Math<double>::copy( data_.m_, m.data_.m_ );
  return *this;
}

inline dvm3_Matrix& dvm3_Matrix::
operator=( double rhs )
{
  vm_M3Math<double>::set( data_.m_, rhs );
  return *this;
}

inline void dvm3_Matrix::
inject_row( int whichrow, double const row[] )
{ vm_M3Math<double>::inject_row( data_.m_, row, whichrow ); }

inline void dvm3_Matrix::
inject_row( int whichrow, dvm3_Vector const& row )
{ vm_M3Math<double>::inject_row( data_.m_, row.data_.x_, whichrow ); }

inline void dvm3_Matrix::
inject_col( int whichcol, double const col[] )
{ vm_M3Math<double>::inject_col( data_.m_, col, whichcol ); }

inline void dvm3_Matrix::
inject_col( int whichcol, dvm3_Vector const& col )
{ vm_M3Math<double>::inject_col( data_.m_, col.data_.x_, whichcol ); }

inline void dvm3_Matrix::
extract_row( int whichrow, double row[] ) const
{ vm_M3Math<double>::extract_row( data_.m_, row, whichrow ); }

inline void dvm3_Matrix::
extract_row( int whichrow, dvm3_Vector& row ) const
{ vm_M3Math<double>::extract_row( data_.m_, row.data_.x_, whichrow ); }

inline void dvm3_Matrix::
extract_col( int whichcol, double col[] ) const
{ vm_M3Math<double>::extract_col( data_.m_, col, whichcol ); }

inline void dvm3_Matrix::
extract_col( int whichcol, dvm3_Vector& col ) const
{ vm_M3Math<double>::extract_col( data_.m_, col.data_.x_, whichcol ); }

inline double const* dvm3_Matrix::
operator[](int i) const
{ return &data_.m_[EColStride*i]; }

inline double* dvm3_Matrix::
operator[](int i)
{ return &data_.m_[EColStride*i]; }

inline double dvm3_Matrix::
operator()(int i, int j) const
{ return *( data_.m_ + EColStride*i + j ); }

inline double& dvm3_Matrix::
operator()(int i, int j)
{ return *( data_.m_ + EColStride*i + j ); }

inline dvm3_Matrix& dvm3_Matrix::operator+()
{ return *this; }

inline dvm3_Matrix& dvm3_Matrix::
operator-()
{ vm_M3Math<double>::negate( data_.m_ );  return *this; }

#define _tpl_OpEQ_(_OPEQ_,_OPNAM_) \
  inline dvm3_Matrix& dvm3_Matrix::operator _OPEQ_ ( dvm3_Matrix const& m ) \
  { vm_M3Math<double>::_OPNAM_( data_.m_, m.data_.m_ ); return *this; } \
  inline dvm3_Matrix& dvm3_Matrix::operator _OPEQ_ ( double r  ) \
  { vm_M3Math<double>::_OPNAM_( data_.m_, r ); return *this; }
  _tpl_OpEQ_(+=,add_eq)
  _tpl_OpEQ_(-=,sub_eq)
  _tpl_OpEQ_(*=,mul_eq)
  _tpl_OpEQ_(/=,div_eq)
#undef _tpl_OpEQ_

#define _tpl_Op_(_OP_,_OPNAM_) \
  inline dvm3_Matrix operator _OP_ ( dvm3_Matrix const& m1,  \
                                     dvm3_Matrix const& m2 ) \
  { dvm3_Matrix res(m1); \
    vm_M3Math<double>::_OPNAM_( res.data_.m_, m1.data_.m_, m2.data_.m_ ); \
    return res; } \
  inline dvm3_Matrix operator _OP_ ( dvm3_Matrix const& m, double r  ) \
  { dvm3_Matrix res(m); \
    vm_M3Math<double>::_OPNAM_( res.data_.m_, m.data_.m_, r ); \
    return res; } \
  inline dvm3_Matrix operator _OP_ ( double r, dvm3_Matrix const& m ) \
  { dvm3_Matrix res(m); \
    vm_M3Math<double>::_OPNAM_( res.data_.m_, r, m.data_.m_ ); \
    return res; }
  _tpl_Op_(+,add)
  _tpl_Op_(-,sub)
  _tpl_Op_(*,mul)
  _tpl_Op_(/,div)
#undef _tpl_Op_

inline void dvm3_Matrix::
dyad_product( dvm3_Vector const& v1, dvm3_Vector const& v2 )
{ vm_M3Math<double>::dyad_product( data_.m_, v1.data_.x_, v2.data_.x_ ); }

inline void dvm3_Matrix::
dyad_product( double const v1[], double const v2[] )
{ vm_M3Math<double>::dyad_product( data_.m_, v1, v2 ); }

inline void dvm3_Matrix::
lincomb( double c1, dvm3_Matrix const& m1,
         double c2, dvm3_Matrix const& m2 )
{ vm_M3Math<double>::lincomb( data_.m_, c1, m1.data_.m_, c2, m2.data_.m_ ); }

inline void dvm3_Matrix::
mvmult( dvm3_Vector& result, dvm3_Vector const& v ) const
{ vm_M3Math<double>::mvmult( result.data_.x_, data_.m_, v.data_.x_ ); }

inline void dvm3_Matrix::
mvmult( double result[], double const v[] ) const
{ vm_M3Math<double>::mvmult( result, data_.m_, v ); }

inline void dvm3_Matrix::
mtvmult( dvm3_Vector& result, dvm3_Vector const& v ) const
{ vm_M3Math<double>::mtvmult( result.data_.x_, data_.m_, v.data_.x_ ); }

inline void dvm3_Matrix::
mtvmult( double result[], double const v[] ) const
{ vm_M3Math<double>::mtvmult( result, data_.m_, v ); }

inline void dvm3_Matrix::
mmult( dvm3_Matrix const& m1, dvm3_Matrix const& m2 )
{ vm_M3Math<double>::mmult( data_.m_, m1.data_.m_, m2.data_.m_ ); }

inline void
dyad_product( dvm3_Matrix& result,
              dvm3_Vector const& v1, dvm3_Vector const& v2 )
{ vm_M3Math<double>::dyad_product( result.data_.m_, 
				    v1.data_.x_, v2.data_.x_ ); }

inline void
dyad_product( dvm3_Matrix& result,
              double const v1[], double const v2[] )
{ vm_M3Math<double>::dyad_product( result.data_.m_, v1, v2 ); }

inline void
lincomb( dvm3_Matrix& result, double c1, dvm3_Matrix const& m1,
                              double c2, dvm3_Matrix const& m2 )
{ vm_M3Math<double>::lincomb( result.data_.m_, 
			       c1, m1.data_.m_, c2, m2.data_.m_ ); }

inline void
mvmult( dvm3_Vector& result, dvm3_Matrix const& m, dvm3_Vector const& v )
{ m.mvmult( result.data_.x_, v.data_.x_ ); }

inline void
mvmult( double result[], dvm3_Matrix const& m, double const v[] )
{ m.mvmult( result, v ); }

inline void
mtvmult( dvm3_Vector& result, dvm3_Matrix const& m, dvm3_Vector const& v )
{ m.mtvmult( result.data_.x_, v.data_.x_ ); }

inline void
mtvmult( double result[], dvm3_Matrix const& m, double const v[] )
{ m.mtvmult( result, v ); }

inline void
mmult( dvm3_Matrix& result, dvm3_Matrix const& m1, dvm3_Matrix const& m2 )
{ result.mmult( m1, m2 ); }

inline std::ostream& dvm3_Matrix::
print_on( std::ostream& os, char const pre[], char const post[] ) const
{ return vm_M3Math<double>::print_on( os, data_.m_, pre, post ); }

inline std::ostream&
operator<<( std::ostream& os, const dvm3_Matrix& m )
{ return m.print_on( os, "[", "]" ); }

inline void dvm3_Matrix::
cprint_on( FILE* of, char const pre[], char const post[] )
{ vm_M3Math<double>::cprint_on( of, data_.m_, pre, post ); }

#endif
