#ifndef RBTREE_H
#define RBTREE_H

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of rbtree
 *
 * rbtree is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rbtree is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>

typedef enum
{
  PRE_ORDER	= 0x01,
  IN_ORDER	= 0x02,
  POST_ORDER	= 0x04
} Visit;

typedef enum { LEFT_TO_RIGHT, RIGHT_TO_LEFT } SiblingOrder;

typedef void * RBTree;
typedef void * RBNode;

#define RBTREE_NULL_CMP ( (int (*)(const void *, const void *)) NULL )
#define RBTREE_NULL_DELETE ( (void (*)(void *)) NULL )

#define RBTREE_DELETE void (*)(void *)

/* *** begin protos *** */
/* cmp.c */
int rbtree_node_cmp_v(const void *dp1, const void *dp2);
int rbtree_node_cmp_s(const void *dp1, const void *dp2);
/* redblack.c */
RBTree rbtree_new(int (*cmp)(const void *, const void *));
void rbtree_delete(RBTree rbtree, void (*nfree)(void *), Visit visit, SiblingOrder sibling_order);
void rbtree_udelete(RBTree rbtree, void (*nfree)(void *, void *), void *udata, Visit visit, SiblingOrder sibling_order);
int rbtree_walk(RBTree rbtree, int (*action)(void *nd, Visit visit, unsigned long level), SiblingOrder sibling_order, Visit visit);
int rbtree_uwalk(RBTree rbtree, int (*action)(void *nd, Visit visit, unsigned long level, void *udata), void *udata, SiblingOrder sibling_order, Visit visit);
int rbtree_traverse(RBTree rbtree, int (*action)(void *nd), SiblingOrder sibling_order);
int rbtree_utraverse(RBTree rbtree, int (*action)(void *nd, void *udata), void *udata, SiblingOrder sibling_order);
unsigned long rbtree_count(RBTree rbtree);
int rbtree_insert(RBTree rbtree, void *data);
int rbtree_insert_dnode(RBTree rbtree, RBNode rbnode);
void *rbtree_destroy(RBTree rbtree, const void *data, int (*cmp)(const void *, const void *));
size_t rbtree_node_size(void);
void *rbtree_destroy_node(RBTree rbtree, RBNode rbnode);
RBNode rbtree_detach_node(RBTree rbtree, RBNode rbnode);
void rbtree_destroy_dnode(RBNode rbnode);
void *rbtree_search(RBTree rbtree, const void *data, int (*cmp)(const void *, const void *));
RBNode rbtree_search_node(RBTree rbtree, const void *data, int (*cmp)(const void *, const void *));
void *rbtree_bnd_search(RBTree rbtree, const void *data, void **prev, void **next, int (*cmp)(const void *, const void *));
RBNode rbtree_bnd_search_node(RBTree rbtree, const void *data, RBNode *prev, RBNode *next, int (*cmp)(const void *, const void *));
void *rbtree_min(RBTree rbtree);
RBNode rbtree_min_node(RBTree rbtree);
void *rbtree_max(RBTree rbtree);
RBNode rbtree_max_node(RBTree rbtree);
RBNode rbtree_next_node(RBTree rbtree, RBNode rbnode, SiblingOrder sib_order);
void *rbtree_node_get_data(RBNode rbnode);
void rbtree_node_put_data(RBNode rbnode, void *data);
void rbtree_join(RBTree tree1, RBTree tree2);
/* *** end protos *** */

#ifdef __cplusplus
}
#endif

#endif	/* RBTREE_H */
