// --8<--8<--8<--8<--
//
// Copyright (C) 2006 Smithsonian Astrophysical Observatory
//
// This file is part of RDB
//
// RDB is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// RDB is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the 
//       Free Software Foundation, Inc. 
//       51 Franklin Street, Fifth Floor
//       Boston, MA  02110-1301, USA
//
// -->8-->8-->8-->8--

#define RDBColumnTmplt_cc

#include <rdbxx/RDBColumnTmplt.h>

/**
 * \param name the column name.
 * \param def the column definition.
 *
 * Allocates space for a single data element of type Tmplt1.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::RDBColumnTmplt( 
  const std::string& name,
  const std::string& def
  )
  : RDBColumn(name,def) {  

  _nelems = 2;
  _data   = new Tmplt0[_nelems];
  _idx    = 0;
  _mine   = true;

}

/**
 * \param rdbcolumntmplt col the RDBColumn object to copy.
 *
 * Makes a shallow copy of the argument.  The two RDBColumn objects
 * share data elements.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::RDBColumnTmplt( 
  const RDBColumnTmplt& col
  ) 
  : RDBColumn(col) {

  _nelems = col._nelems;
  _data   = new Tmplt0[_nelems];
  _idx    = col._idx;
  _mine   = true;

}

/**
 * Responsible for freeing data element memory allocated by the
 * RDBColumn object.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::~RDBColumnTmplt( 
  void 
  ) {

  cleanup( );

}

/**
 * \param col the RDBColumn object to copy.
 *
 * Makes a shallow copy of the argument.  The two RDBColumn objects
 * share data elements.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumn& 
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::operator=( 
  const RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>& col 
  ) {

  if ( this != &col ) {
    RDBColumn::operator=( col );

    cleanup( );

    _nelems = col._nelems;
    _data   = col._data;
    _idx    = col._idx;
    _mine   = false;

    if ( _group && 2 > _nelems ) 
      throw( RDBErr( "RDBColumnTmplt::operator=(RDBColumnTmplt&):  user supplied column must provide >= 2 data elements" ) );      

  }

  return *this;

}

/**
 * \param data
 *
 * Assigns the value to the current RDBColumn data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumn& 
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::operator=( 
  const Tmplt0& data 
  ) {

  try { 
    convert( data, _data[_idx] );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::operator=(Tmplt0&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::operator=(Tmplt0&):  unexpected exception caught" ) );
    
  }

  return *this;

}

/**
 * \param data
 *
 * \exception RDBErr error if the user attempts to convert non-numeric string data to a numeric column.
 * \exception RDBErr error if the user attempts to convert string data representing a floating point number to an integer column.
 *
 * Assigns the value to the current RDBColumn data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumn& 
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::operator=( 
  const Tmplt1& data 
  ) {

  try { 
    convert( data, _data[_idx] );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::operator=(Tmplt1&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::operator=(Tmplt1&):  unexpected exception caught" ) );
    
  }

  return *this;

}

/**
 * \param data
 *
 * \exception RDBErr error if the user attempts to convert non-numeric string data to a numeric column.
 * \exception RDBErr error if the user attempts to convert string data representing a floating point number to an integer column.
 *
 * Assigns the value to the current RDBColumn data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
RDBColumn& 
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::operator=( 
  const Tmplt2& data 
  ) {

  try {
    convert( data, _data[_idx] );
    
  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::operator=(Tmplt2&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::operator=(Tmplt2&):  unexpected exception caught" ) );
    
  }

  return *this;

}

/**
 * Advances the automatic index for the data elements by one.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::advanceIdx(
  void
  ) {

  if ( _nelems == ++_idx ) {
    _idx  = 0;

  }
}

/**
 * Rewinds the automatic index for the data elements to the first element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::rewind(
  void
  ) {

  _idx = 0;

}

/**
 * \param group incidcates whether or not this is a group column.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::setGroup( 
  bool group
  ) {
  
  RDBColumn::setGroup( group );
  if ( _mine ) {
    if ( 2 > _nelems ) {
      Tmplt0* tmp = _data;
      _nelems     = 2;
      _data       = new Tmplt0[_nelems];
      _data[0]    = tmp[_idx];
      _data[1]    = tmp[_idx];
      delete [] tmp;    
    }
  } else if ( 2 > _nelems ) {
    throw( RDBErr( "RDBColumnTmplt::setGroup(bool):  user supplied column must provide >= 2 data elements" ) );

  }
}

/**
 * Sets the group for this object to the current data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::setGroupValue(
  void
  ) {

  if ( _group ) {
    _data[(_idx+1)%_nelems] = _groupvalue;
    _groupvalue = _data[_idx];
  }

}

/**
 * \return True if the data element is in a new group.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
int
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::newGroup(
  void
  ) {

  if ( _group ) {
    if ( _initgroup ) {
      _initgroup = false;
      _data[_idx] = _groupvalue;
      _groupvalue = _data[(_idx+1)%_nelems];
      return RDBColumn::CBOG;

    } else {
      int status = RDBColumn::CMOG;
      
      if ( _data[_idx] != _groupvalue )
	status |= RDBColumn::CBOG;

      if ( _data[_idx] != _data[(_idx+1)%_nelems] )
	status |= RDBColumn::CEOG;

      _groupvalue = _data[_idx];

      return status;
    }
  }

  return RDBColumn::CMOG;
  
  /*
  if ( _group ) {
    if ( _initgroup ) {
      _initgroup  = false;
      _data[(_idx+1)%_nelems] = _groupvalue;
      _groupvalue = _data[_idx];
      return RDBColumn::CBOG;

    } else if ( _data[_idx] != _groupvalue ) {
      _data[(_idx+1)%_nelems] = _groupvalue;
      _groupvalue = _data[_idx];
      return RDBColumn::CBOG;

    }
  }

  return RDBColumn::CMOG;
  */
}

/**
 * \param data
 *
 * \return True upon successful conversion, false otherwise.
 *
 * Assigns the value to the current RDBColumn data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
bool
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::setData(
  const Tmplt0& data
  ) {

  try {
    convert( data, _data[_idx] );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::setData(Tmplt0&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::setData(Tmplt0&):  unexpected exception caught" ) );
    
  }

  return 0 == _errno;

}

/**
 * \param data
 *
 * \exception RDBErr error if the user attempts to convert non-numeric string data to a numeric column.
 * \exception RDBErr error if the user attempts to convert string data representing a floating point number to an integer column.
 *
 * \return True upon successful conversion, false otherwise.
 *
 * Assigns the value to the current RDBColumn data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
bool
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::setData(
  const Tmplt1& data
  ) {

  try { 
    convert( data, _data[_idx] );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::setData(Tmplt1&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::setData(Tmplt1&):  unexpected exception caught" ) );
    
  }

  return 0 == _errno;

}

/**
 * \param data
 *
 * \exception RDBErr error if the user attempts to convert non-numeric string data to a numeric column.
 * \exception RDBErr error if the user attempts to convert string data representing a floating point number to an integer column.
 *
 * \return True upon successful conversion, false otherwise.
 *
 * Assigns the value to the current RDBColumn data element.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
bool
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::setData(
  const Tmplt2& data
  ) {

  try { 
    convert( data, _data[_idx] );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::setData(Tmplt2&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::setData(Tmplt2&):  unexpected exception caught" ) );
    
  }

  return 0 == _errno;

}

/**
 * \param data pointer to a data element of type Tmplt0.
 * \param nelems number of data elements in the array pointed by data.
 *
 * This method associates user allocated memory pointed to by data
 * with this RDBColumn object.  The used is responsible for freeing
 * the memory pointed to by data after the RDBColumn is destroyed.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::mapData(
  Tmplt0 data[], 
  const size_t nelems
  ) {

  cleanup( );

  _nelems     = nelems;
  _data       = data;
  _idx        = _idx % _nelems;
  _mine       = false;

  if ( _group && 2 > _nelems ) 
    throw( RDBErr( "RDBColumnTmplt::mapData:  user supplied data must provide >= 2 data elements" ) );

}

/**
 * \return Pointer to the current data element.
 *
 * This method returns a pointer to the current data element.
 * Modifications to the data returned are evident within this object.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void*
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getData( 
  void 
  ) { 

  return ((void *) &_data[_idx]);

}

/**
 * \param data assigned the value of the current data element in this object.
 *
 * \return True if the conversion was successful, false otherwise.
 *
 * Assigns the value of the current RDBColumn data element to the arguement.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
bool
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getData(
  Tmplt0& data
  ) {

  try { 
    convert( _data[_idx], data );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::getData(Tmplt0&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::getData(Tmplt0&):  unexpected exception caught" ) );
    
  }

  return 0 == _errno;

}

/**
 * \param data assigned the value of the current data element in this object.
 *
 * \exception RDBErr error if the user attempts to convert a string column with non-numeric data to a numeric arguemnt.
 * \exception RDBErr error if the user attempts to convert a string column representing a floating point number to an integer argument.
 *
 * \return True if the conversion was successful, false otherwise.
 *
 * Assigns the value of the current RDBColumn data element to the arguement.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
bool
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getData(
  Tmplt1& data
  ) {

  try { 
    convert( _data[_idx], data );
    
  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::getData(Tmplt1&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::getData(Tmplt1&):  unexpected exception caught" ) );
    
  }

  return 0 == _errno;

}

/**
 * \param data assigned the value of the current data element in this object.
 *
 * \exception RDBErr error if the user attempts to convert a string column with non-numeric data to a numeric arguemnt.
 * \exception RDBErr error if the user attempts to convert a string column representing a floating point number to an integer argument.
 *
 * \return True if the conversion was successful, false otherwise.
 *
 * Assigns the value of the current RDBColumn data element to the arguement.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
bool
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getData(
  Tmplt2& data
  ) {

  try {
    convert( _data[_idx], data );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::getData(Tmplt2&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::getData(Tmplt2&):  unexpected exception caught" ) );
    
  }

  return 0 == _errno;

}

/**
 * \exception RDBErr error if the user attempts to convert a string column with non-numeric data to a numeric arguemnt.
 *
 * \return The data element as a double.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
double
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getDataDouble(
  void
  ) {

  double odata;
  try { 
    convert( _data[_idx], odata );
    
  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::getDataDouble(void):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::getDataDouble(void):  unexpected exception caught" ) );
    
  }

  return odata;

}

/**
 * \exception RDBErr error if the user attempts to convert a string column with non-numeric data to a numeric arguemnt.
 * \exception RDBErr error if the user attempts to convert a string column representing a floating point number to an integer argument.
 *
 * \return The data element as a long.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
long
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getDataLong(
  void
  ) {

  long odata;
  try {
    convert( _data[_idx], odata );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::getDataLong(void):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::getDataLong(void):  unexpected exception caught" ) );
    
  }

  return odata;

}

/**
 * \return The data element as a string.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
std::string
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::getDataString(
  void
  ) {
 
  std::string odata;  
  try { 
    convert( _data[_idx], odata );
    
  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::getDataString(void):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::getDataString(void):  unexpected exception caught" ) );
    
  }

  return odata;

}

/**
 * \param is input stream.
 *
 * \return The input stream.
 *
 * Called by the RDBColumn stream extraction operator.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
std::istream&
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::read( 
  std::istream& is
  )  {

  try {
    extract( is, _data[_idx] );

  } catch ( RDBErr& rdberr ) {
    rdberr.set_message( "RDBColumnTmplt::read(std::istream&):  " );
    throw( rdberr );
    
  } catch ( ... ) { 
    throw( RDBErr( "RDBColumnTmplt::read(std::istream&):  unexpected exception caught" ) );
    
  }

  return is;

}

/**
 * \param os output stream.
 *
 * \return The output stream.
 *
 * Called by the RDBColumn stream insertion operator.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
std::ostream&
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::write(
  std::ostream& os
  ) const {

  return insert( os, _data[_idx] );

}

/**
 * Frees memory allocated by this object.
 *
 */
template<class Tmplt0,class Tmplt1,class Tmplt2>
void
RDBColumnTmplt<Tmplt0,Tmplt1,Tmplt2>::cleanup(
  void
  ) {

  if ( _mine && _data ) {
    delete [] _data;

  }
}
