#ifndef rl_Multilayer_h_INCLUDED
#define rl_Multilayer_h_INCLUDED

// File:   rl_Multilayer.h
// Author: Terry Gaetz

/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006, 2007 Smithsonian Astrophysical Observatory
 *
 * This file is part of rl_raylib
 *
 * rl_raylib is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * rl_raylib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

/****************************************************************************
 * encapsulate the multilayer surface properties:
 *    . multilayer properties
 *    . complex reflection coefficient
 *
 * History
 *--------
 * 1.0.1 2004-Dec-07  tjg  simplify/remove include guards
 * 0.0.0 1998-Jun-24  tjg  original version
 */

#include <cstdlib>                          // size_t
#include <rl_raylib/rl_Traits.h>            // rl_Traits::EInterpMode complex
#include <rl_raylib/rl_DielectricLayer.h>   // rl_Traits::EInterpMode complex

class rl_ReflectionCoefPOD;                 // forward declaration

//########################################################################
// rl_Multilayer
//########################################################################

/** 
 * \class rl_Multilayer
 *
 * A class encapsulating reflection of a ray from a multilayer
 * surface.  Given the energy and the sine of the graze angle, 
 * reflection coefficient can be evaluated.
 */

class rl_Multilayer
{
private:

  /// Prohibited method; declared private and not defined.
  rl_Multilayer();
  /// Prohibited method; declared private and not defined.
  rl_Multilayer( rl_Multilayer const& );
  /// Prohibited method; declared private and not defined.
  rl_Multilayer& operator=( rl_Multilayer const& );

protected:

  /// number of multilayers
  int                   num_layers_;
  /// the top (vacuum) layer 
  rl_DielectricLayer    vacuum_;
  /// array of dielectric layers
  rl_DielectricLayer*   layer_;
  /// do we own the rl_DielectricLayer array?
  rl_Traits::Bool       own_data_;

public:

  /// interpolation mode for dielectric data
  typedef rl_Traits::EInterpMode EInterpMode;

  /**
   * Destructor.
   * 
   * Frees up rl_DielectricLayer array if own_data_ is rl_Traits::True.
   */
  virtual ~rl_Multilayer();

  /**
   * Construct multilayer from num_layers individual layers.
   *
   * @param num_layers number of layers to construct.
   * @param layers     pointer to array of layer data
   * @param adopt_data boolean flag
   *    - if rl_Traits::True, rl_Multilayer assumes responsibility for 
   *      deleting the rl_DielectricLayer array;
   *    - if rl_Traits::False, it is the caller's responsibility.
   */
  rl_Multilayer( int                 num_layers, 
                 rl_DielectricLayer* layers,
                 rl_Traits::Bool     adopt_data = rl_Traits::True );

  /**
   * Initialize multilayer from num_layers individual layers.
   *
   * @param num_layers number of layers to construct.
   * @param layers     pointer to array of layer data
   * @param adopt_data boolean flag
   *    - if rl_Traits::True, rl_Multilayer assumes responsibility for 
   *      deleting the rl_DielectricLayer array;
   *    - if rl_Traits::False, it is the caller's responsibility.
   */

  void 
  init( int                 num_layers, 
        rl_DielectricLayer* layers,
        rl_Traits::Bool     adopt_data = rl_Traits::True );

  /**
   * Evaluate the multilayer reflection coefficients
   *
   * @return 0 if successful, the number of the invalid layer if not.
   *
   * @param rfl    the computed reflection coefficient.
   * @param energy ray energy.
   * @param sg     sine of the graze angle between the ray and the surface.
   */
  int
  multilayer_reflect_coef( 
                           rl_ReflectionCoefPOD& rfl, 
                                            //out: reflection coefficient
                           double energy,   // in:  energy of ray
                           double sg        // in:  sine graze angle
                         );

  /**
   * Evaluate the multilayer reflectivity (assuming unpolarized rays)
   * 
   * @return 0 if successful, the number of the invalid layer if not.
   *
   * @param rfl multilayer reflectivity.
   * 
   * @param energy ray energy.
   * @param sg     sine of the graze angle between the ray and the surface.
   * @param polarization_factor polarization factor; it must be
   *        a value between -1 and 1.  The polarization factor is 
   *        related to parallel (p) and perpendicular (s) polarization by:
   *  \f[
   *      \mathtt{polarization\_factor} = {(I_\perp - I_\parallel) 
   *                                     \over 
   *                                     (I_\perp + I_\parallel)} 
   *  \f]
   * or
   *  \f[
   *      \mathtt{polarization\_factor} = {(I_\mathit{s} - I_\mathit{p}) 
   *                                     \over 
   *                                     (I_\mathit{s} + I_\mathit{p})} 
   *  \f]
   * where \f$I_\perp\f$ and \f$I_\parallel\f$ are the perpendicular
   * and parallel E-field \em intensities, respectively.  Thus,
   *  - -1: pure parallel (p) polarization.
   *  - 0: completely unpolarized.
   *  - +1: pure perpendicular (s) polarization.
   */
  int 
  multilayer_reflectivity( double& rfl,    // out: multilayer reflectivity
                           double  energy, // in:  energy of ray
                           double  sg,     // in:  sine graze angle
                           double  polarization_factor = 0.0 // in:
                         );

  /**
   * @return const reference to layer layer_no
   */
  rl_DielectricLayer const& layer( int layer_no ) const;

  /**
   * @return number of layers
   */
  int num_layers() const;

  /**
   * Dump information about layer layer_no to stream os.
   *
   * @return reference to stream os.
   *
   * @param os output stream
   * @param layer_no layer number to be dumped
   * @param pre optional prefix string
   * @param pst optional postfix string
   */
  std::ostream& dump_on( std::ostream& os, int layer_no,
                                           char const pre[] = "",
                                           char const pst[] = "" ) const;

  /**
   * Dump information about layer layer_no to output FILE*.
   *
   * @param of output FILE*
   * @param layer_no layer number to be dumped
   * @param pre optional prefix string
   * @param pst optional postfix string
   */
  void cdump_on( std::FILE* of, int layer_no,
                                char const pre[] = "",
                                char const pst[] = "" ) const;
};

//########################################################################
//########################################################################
//
//    #    #    #  #          #    #    #  ######   ####
//    #    ##   #  #          #    ##   #  #       #
//    #    # #  #  #          #    # #  #  #####    ####
//    #    #  # #  #          #    #  # #  #            #
//    #    #   ##  #          #    #   ##  #       #    #
//    #    #    #  ######     #    #    #  ######   ####
//
//########################################################################
//########################################################################


//=========================================================================
// accessors
inline int rl_Multilayer::
num_layers() const
{
  return num_layers_;
}

// rl_Multilayer_h_INCLUDED
#endif
