import os
from abc import ABCMeta, abstractmethod
from enum import Enum
from inspect import Parameter as SignatureParameter, Signature
from pathlib import Path
from typing import Type
from weakref import WeakKeyDictionary


class ConfigurationError(AttributeError):
    pass


class Parameter:
    def __init__(self, datatype, default=None, name=None, validator=None):
        self.datatype = datatype
        self.default = default
        self.name = name
        self.validator = validator
        self.registry = WeakKeyDictionary()

    def __get__(self, instance, instance_class):
        if instance is None:
            return self
        return self.registry[instance]

    def __set__(self, instance, value):
        try:
            value = self.datatype(value)
        except (ValueError, TypeError):
            raise ValueError(f"Could not cast '{value}' to {self.datatype} (Parameter \'{self.name}\')")
        if self.validator:
            self.validator.validate(value)
        self.registry[instance] = value

    def __set_name__(self, owner: "Configuration", name):
        if not hasattr(owner, '__parameters__'):
            owner.__parameters__ = {}
        owner.__parameters__[name] = self
        self.name = name

    def __str__(self):
        return f'{self.__class__.__name__}(name={self.name}, datatype={self.datatype}, default={self.default})'

    def __repr__(self):
        return str(self)

    def __eq__(self, other):
        return repr(self) == repr(other)

    def __hash__(self):
        return hash(repr(self))

    def describe(self):
        return f'Parameter {self.name}: default={self.default}, type={self.datatype}'


class Validator(metaclass=ABCMeta):
    def __init__(self, parameter):
        self.parameter = parameter

    @abstractmethod
    def validate(self, value):
        raise NotImplementedError()


class PositiveValidator(Validator):
    def validate(self, value):
        if value <= 0:
            raise ValueError(f'Parameter {self.parameter.name} must be positive')


class NonNegativeValidator(Validator):
    def validate(self, value):
        if value < 0:
            raise ValueError(f'Parameter {self.parameter.name} must be non negative')


class String(Parameter):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs, datatype=str)


class Integer(Parameter):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs, datatype=int)


class Float(Parameter):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs, datatype=float)


class NormalizedValidator(Validator):
    def validate(self, value):
        if value < 0 or value > 1:
            raise ValueError(f'Parameter {self.parameter.name} must be between 0 and 1')


class Normalized(Parameter):
    def __init__(self, default=None):
        super().__init__(datatype=float, default=default, validator=NormalizedValidator(self))


class Decorator(Parameter):
    def __init__(self, parameter, validator):
        super().__init__(datatype=parameter.datatype, default=parameter.default, name=parameter.name,
                         validator=validator)
        self.parameter = parameter

    def __set_name__(self, owner, name):
        super().__set_name__(owner, name)
        self.parameter.name = name

    def __str__(self):
        return f'{self.__class__.__name__}({str(self.parameter)})'


class Positive(Decorator):
    def __init__(self, parameter):
        super().__init__(parameter, validator=PositiveValidator(self))


class NonNegative(Decorator):
    def __init__(self, parameter):
        super().__init__(parameter, validator=NonNegativeValidator(self))


class File(Parameter):
    def __init__(self, required=True, output=False, placeholders=None):
        self.required = required
        default = '' if not required else None
        self.placeholders = placeholders
        super().__init__(validator=FileValidator(self, output), datatype=Path, default=default)

    def describe(self):
        result = f'Parameter {self.name}: default={self.default}, type=File'
        if self.placeholders is not None:
            result += f' template with placeholders {self.placeholders}'
        return result

    def require(self, value):
        return self.validator.validate(value, require=True)


def boolean_converter(value):
    if value in ('0', 'False', 'false', 'FALSE', 'no', 'NO', 'No', False):
        return False
    else:
        return True


class Boolean(Parameter):
    def __init__(self, default=True):
        super().__init__(validator=NullValidator(self), datatype=boolean_converter, default=default)

    def describe(self):
        return f'Parameter {self.name}: default={self.default}, type=Boolean'


class NullValidator(Validator):
    def validate(self, value):
        return


class FileValidator(Validator):
    def __init__(self, parameter, output=False):
        super().__init__(parameter)
        self.output = output

    def validate(self, value: Path, require=False):
        if self.output:
            self._validate_output(value)
        else:
            self._validate_input(value, require=require)

    def _validate_input(self, value: Path, require=False):
        if (self.parameter.required or require) and (not value.exists() or not value.is_file()):
            raise ValueError(f'Path {value} does not exist or is not a file.')

    def _validate_output(self, value: Path):
        if not value.name:
            raise ValueError(f'The name of an output file cannot be empty (Parameter \'{self.parameter.name}\')')
        if not value.parent.exists() or not os.access(value.parent, os.W_OK):
            raise ValueError(f'Path {value} is in a directory that does not exist '
                             f'or that is not writeable')
        if value.name and hasattr(self.parameter, 'placeholders') and self.parameter.placeholders is not None:
            for ph in self.parameter.placeholders:
                if ph not in value.name:
                    raise ValueError(f'Filename template is missing at least one of the placeholders in '
                                     f'{self.parameter.placeholders}')


class ListOfFiles(Parameter):
    def __init__(self):
        super().__init__(datatype=None, validator=FileValidator(File()))

    def __set__(self, instance, value):
        import stk
        if not value:
            raise ValueError(f'A file list instance cannot be empty (Parameter \'{self.name}\')')
        if self._should_set(value):
            files = [Path(file) for file in stk.build(value)]
            for file in files:
                self.validator.validate(file)
        else:
            files = []
        self.registry[instance] = files

    def values_description(self, instance):
        return ",\n\t".join(str(file) for file in self.registry[instance])

    def describe(self):
        return f'Parameter {self.name}: default={self.default}, type=ListOfFiles'

    def _should_set(self, value):
        import stk
        if not value.startswith('@'):
            return True
        if not Path(value.replace('@', '')).exists():
            return False
        try:
            test = stk.build(value)
        except:
            return False
        return True


class DirectoryValidator(Validator):
    def validate(self, value):
        if not os.path.isdir(value.parent):
            raise ConfigurationError("The specified directory path does not exist and neither does its parent")
        if value.exists() and not value.is_dir():
            raise ConfigurationError("The specified directory path is not a directory")
        if not value.exists():
            value.mkdir()


class Directory(Parameter):
    def __init__(self):
        super().__init__(datatype=Path, validator=DirectoryValidator(self))


class Category(Parameter):
    def __init__(self, enum: Type[Enum], default=None):
        super().__init__(datatype=enum, default=default)

    def describe(self):
        return f'Parameter {self.name}: default={self.default.value}, type=Enumeration - accepted strings ' \
               f'{self.acceptable_values()}'

    def acceptable_values(self):
        return [e.value for e in self.datatype]


class Configuration:
    def __new__(cls: Type["Configuration"], *args, **kwargs):
        obj = object.__new__(cls)
        if not hasattr(obj, '__parameters__'):
            obj.__parameters__ = {}
        signature = ConfigurationSignature(cls)
        signature.bind(obj, *args, **kwargs)
        obj.validate()
        return obj

    def validate(self):
        pass

    def __contains__(self, item):
        return item in self.__parameters__

    def __str__(self):
        result = "Configuration:\n"
        for name, parameter in self.__parameters__.items():
            if hasattr(parameter, 'values_description'):
                result += f"\tParameter {name}: {parameter.values_description(self)}\n"
            else:
                result += f"\tParameter {name}: {getattr(self, name)}\n"
        return result

    @classmethod
    def is_file_set(cls, file):
        return file != Path('')

    @classmethod
    def describe(cls):
        result = ""
        for parameter in cls.__parameters__.values():
            result += parameter.describe() + '\n'
        return result


class ConfigurationSignature:
    def __init__(self, cls: Type[Configuration]):
        self.cls = cls
        self.signature = Signature(self._make_parameters())

    def bind(self, obj, *args, **kwargs):
        bound = self.signature.bind(*args, **kwargs)
        bound.apply_defaults()
        for k, v in bound.arguments.items():
            setattr(obj, k, v)

    def _make_parameters(self):
        parameters = []
        kind = SignatureParameter.POSITIONAL_OR_KEYWORD
        for name, parameter in self.cls.__parameters__.items():
            if parameter.default is None:
                parameters.append(SignatureParameter(name, kind))
            else:
                parameters.append(SignatureParameter(name, kind, default=parameter.default))
        return parameters
