import pytest
from unittest.mock import patch
from ..extract_keywords import extract_fits_keywords
from concurrent.futures import Executor, Future


class SerialExecutor(Executor):
    """
    A simple executor implementation that runs tasks synchronously for testing purposes.
    """
    def submit(self, fn, *args, **kwargs):
        future = Future()
        try:
            result = fn(*args, **kwargs)
            future.set_result(result)
        except Exception as e:
            future.set_exception(e)
        return future


@pytest.fixture
def mock_fits_files():
    """
    Mock the return values from reading FITS files for testing.
    """
    header_values = [
        {'DATE-OBS': '2023-07-01', 'EXPTIME': '1200', 'INSTRUMENT': 'Camera1'},
        {'DATE-OBS': '2023-07-02', 'EXPTIME': '1500', 'INSTRUMENT': 'Camera2'}
    ]
    files = ['file1.fits', 'file2.fits']
    return files, header_values


@patch('ciao.tools.extract_ids.extract_keywords.read_file')
@patch('ciao.tools.extract_ids.extract_keywords.get_keyval')
def test_extract_keywords_from_file(mock_key_val, _, mock_fits_files):
    """
    Test the function to extract keywords from a single file.
    """
    file, headers = mock_fits_files

    def get_keyval(_, key):
        return headers[0][key]

    mock_key_val.side_effect = get_keyval

    # Execute with the SerialExecutor
    executor = SerialExecutor()
    results = extract_fits_keywords([file[0]], ['DATE-OBS', 'EXPTIME'], executor)

    # Assertions
    assert file[0] in results
    assert results[file[0]]['DATE-OBS'] == '2023-07-01'
    assert results[file[0]]['EXPTIME'] == "1200"


def test_executor_with_multiple_files(mock_fits_files):
    """
    Test processing multiple files with the serial executor.
    """
    files, headers = mock_fits_files
    with patch('ciao.tools.extract_ids.extract_keywords.extract_keywords_from_file', side_effect=[
        (files[0], headers[0]),
        (files[1], headers[1])
    ]):
        executor = SerialExecutor()
        results = extract_fits_keywords(files, ['DATE-OBS', 'EXPTIME'], executor)

        # Assertions
        assert files[0] in results
        assert files[1] in results
        assert results[files[0]]['DATE-OBS'] == '2023-07-01'
        assert results[files[1]]['DATE-OBS'] == '2023-07-02'
