#!/opt/conda/envs/ciao-4.17.0/bin/python3
# 
#  Copyright (C) 2006-2007,2016  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

from __future__ import print_function

from pytransform import *
from numpy import array
import numpy
from os import environ


#  Setup input/output directories
#
testin  = environ.get('TESTIN')
if testin == None :
  indir = environ.get('PWD')
else :
  indir = testin + '/test_transform'

testout = environ.get('TESTOUT')
if testout == None :
  outdir = environ.get('PWD')
else :
  outdir = testout + '/transform'


print("\n************************************************************")
print("                      LINEAR2DTransform                     ")
print("************************************************************")
#
# Wrapped and running
#   LINEAR2DTransform();
#   LINEAR2DTransform(string in_name);
#   LINEAR2DTransform(string in_name, list<TransParam*> in_params);
#   LINEAR2DTransform(string in_name, LINEAR2DTransform&);
#
#   string get_name();
#   void   set_name(string in_name);
#
#   void set_parameter_list(list<TransParam*> in_params);
#   list<TransParam*> get_parameter_list();
#   void print_parameter_list();
#
#   trErrCode  set_parameter(TransParam* in_param);
#   TransParam* get_parameter(string in_name);
#   void*      get_parameter_value(string in_name);
#
#   void* apply(void* in_array, dmDataType in_datatype);
#   void* invert(void* in_array, dmDataType in_datatype);
#
# Ignore in Python..
#   void clear();
# 

# ----------------------------------------------------------------------  


# ----------------------------------------------------------------------  
print("\nCreate LINEAR2D Transform object")
T1 = LINEAR2DTransform()
print("  T1 = LINEAR2DTransform()... OK")

print("\nDisplay Default Parameter Info")
print("  T1.print_parameter_list()...")
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nCreate LINEAR2D Transform object with name.")
T2 = LINEAR2DTransform("T2")
print("  T2 = LINEAR2DTransform('T2')... OK")

print("\nDisplay Default Parameter Info")
print("  T2.print_parameter_list()...")
print(T2.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nName handling")

name = T2.get_name()
print("  T2.get_name()            => ", name)

T1.set_name("T1")
print("  T1.set_name('T1')")
print("  T1.get_name()            => ", T1.get_name())

# ----------------------------------------------------------------------  

print("\nDisplay Parameter Info")
print("  T1.print_parameter_list()...")
print(T1.print_parameter_list())

print("\n  T2.print_parameter_list()...")
print(T2.print_parameter_list())


print("\nGet Parameter List from Transform")
print("  Params = T1.get_parameter_list()")
Params = T1.get_parameter_list()
print("   Params list of type ", type(Params).__name__)
print("   Params list length ", len(Params))
print("   Params list elements of type ", type(Params[1]).__name__)
print("   Display List Elements:")
for ii in range(0, len(Params)) :
  print((Params[ii]).show())

print("\nDisplay Parameter Info Directly")
print("  T1.print_parameter_list()... ")
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  

print("\nGet Specific Parameter from Transform")
print("  rotParam = T1.get_parameter('ROTATION')")
rotParam = T1.get_parameter("ROTATION")
print(rotParam.show())

print("\n  rotParam.set_value(5.0)")
print("   T1.print_parameter_list()")
rotParam.set_value(5.0)
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  


print("\nGet Specific Parameter VALUE from Transform")
rota = T1.get_parameter_value("ROTATION")
print("  T1.get_parameter_value('ROTATION') => ", rota) 

rota = T1.get_parameter_value("BLAH")
print("  T1.get_parameter_value('BLAH') => ", rota) 

# ---------------------------------------------------------------------- 

print("\nCreate a Transform using input Parameter list")
print("  - Make a Parameter list.")
crpix = TransParam("SCALE",[-0.136667, 0.136667], "Scale")
print(crpix.show())
crval = TransParam("ROTATION", 0.0, "Rotation Angle")
print(crval.show())
cdelt = TransParam("OFFSET",[4145.697897, 4098.235660], "Origin Offset")
print(cdelt.show())
Params = list([crpix, crval,cdelt])
print("  TransParam list of type ", type(Params).__name__)

print("  - Use it in constructor.")
T3 = LINEAR2DTransform("T3", Params )
print("  T3 = LINEAR2DTransform('T3', Params )... OK")
print("  T3.print_parameter_list()... NOTE: Only values change from default.")
print(T3.print_parameter_list())

# ----------------------------------------------------------------------  

print("- Set input sky(X,Y) data array") 
sky = array([[0984.5, 1020.5],
             [0994.5, 1021.5],
             [1004.5, 1022.5],
             [1014.5, 1023.5],
             [1024.5, 1024.5],
             [1034.5, 1025.5],
             [1044.5, 1026.5],
             [1054.5, 1027.5],
             [1064.5, 1028.5],
             [1074.5, 1029.5]], dtype=numpy.double)


print("- Apply Transform") 
cel = T3.apply(sky)

print("  Input array length.. ", len(sky))
print("  Input array ndim.. ", sky.ndim)
print("  Input array shape.. ", sky.shape)
print("  Input array dtype.. ", sky.dtype)
print("  Output cel values are type... ", type(cel).__name__)
print("  Output array length.. ", len(cel))
print("  Output array ndim.. ", cel.ndim)
print("  Output array shape.. ", cel.shape)
print("  Output array dtype.. ", cel.dtype)

print("  X    ,   Y     ->     RA         ,     DEC ")
for ii in range(0, len(sky)) :
  print(sky[ii][0], ",", sky[ii][1], " -> ", cel[ii][0], ",", cel[ii][1])

# ----------------------------------------------------------------------  
print("\nApply Inverse Transform to data")
print("- Input data array is transformed cel(RA,DEC) data")
print("- Apply Inverse Transform") 
xyout = T3.invert(cel)

print("  Input array length.. ", len(cel))
print("  Input array ndim.. ", cel.ndim)
print("  Input array shape.. ", cel.shape)
print("  Input array dtype.. ", cel.dtype)
print("  Output cel values are type... ", type(xyout).__name__)
print("  Output array length.. ", len(xyout))
print("  Output array ndim.. ", xyout.ndim)
print("  Output array shape.. ", xyout.shape)
print("  Output array dtype.. ", xyout.dtype)


print("   RA         ,     DEC         ->    X    ,   Y  ")
for ii in range(0, len(cel)) :
  print(cel[ii][0], ",", cel[ii][1], " -> ", xyout[ii][0], ",", xyout[ii][1])


# ----------------------------------------------------------------------  

print("\nSet Transform Parameters from Transform Matrix")

T4 = LINEAR2DTransform("T4")
print(T4.print_parameter_list())

matrix = T3.get_transform_matrix()

print("\nCall set_transform_matrix()")
T4.set_transform_matrix(matrix)
print(T4.print_parameter_list())

# ----------------------------------------------------------------------  

print("\nCopy T4 Transform into a new Transform called T5")

T5 = T4.copy()
T5.set_name("T5")
print(T5.print_parameter_list())

print("\nUpdate the SCALE parameter in T5")
scale = T5.get_parameter("SCALE")
scale.set_value((0.22222, -0.22222))

print("\nShow T5")
print(T5.print_parameter_list())

print("\nShow T4")
print(T4.print_parameter_list())

matrix = T5.get_transform_matrix()
print("\nPrint T5 transform matrix\n", matrix)


# ----------------------------------------------------------------------  

print("\nDone.\n")
