-- saotrace.config.raygen.scripts.entrance_aperture script to
-- create entrance apertures from a list in a database.

local assert   = assert
local require  = require
local type     = type
local error    = error
local tonumber = tonumber
local string   = string

local raygen   = require( 'saotrace.raygen' )
local ideal    = require( 'saotrace.baffles.ideal' )

local rs       = require( 'saotrace.raygen.ea.utils.ring_n_spoke' )
local annulus  = require( 'saotrace.raygen.ea').annulus
local pinhole  = require( 'saotrace.raygen.ea').pinhole

local vobj     = require( 'saotrace.raygen.validate' ).vobj:new()
local tables   = require( 'saotrace.suplib.tables' )

local M = {}
setfenv( 1, M )

local vspec = {
   file  = { type = 'string' },
   shell = { type = 'posnum' },
   ea  = {
      default = 'annulus',
      type = {
	 default = { enum = { 'annulus', 'ringnspoke' } },
	 ringnspoke = { vtable = {
			   type = { enum = 'ringnspoke' },
			   ipad   = { type = 'number', default = 0   },
			   opad   = { type = 'number', default = 0   },
			},
		     },
      },
   }
}

-- deep copies of the arguments we'll be passing to rs.ringNspoke
local vspec_rs = tables.copy(
   tables.extract( rs.vspec.ringNspoke,
		   { 'nrings', 'nspokes', 'trot', 'tmin', 'tmax', 'dist' } ),
   true )

-- set some defaults
vspec_rs.nrings.default = 1
vspec_rs.nspokes.default = 12

-- and merge with the spec
tables.merge( vspec.ea.type.ringnspoke.vtable, vspec_rs )

-- create an ideal entrance aperture from the mirror geometry
function entrance_aperture( ... )

   local ok, args = vobj:validate( vspec, ... )
   assert( ok, args )

   local ea = args.ea
   args.ea = nil

   -- rewrite args to work with ideal.create
   args.db = args.file
   args.file = nil
   args.cols = { x = 'x0', y = 'y0', z = 'z_f', r = 'rho_f' }
   args.optic = 'p'

   local inner, outer = ideal.create( args )

   if type(ea) == 'string' then

      if ea == 'annulus' then

	 annulus( 'ideal ea', { x = outer.x, y = outer.y, z = outer.z,
				ri = inner.r, ro = outer.r } )

      elseif ea == 'ringnspoke' then

	 ok, ea = vobj:validate( vspec.ea.type.ringnspoke.vtable,
				 { type = 'ringnspoke' } )

	 assert( ok, ea )

      else

	 error( "unknown entrance aperture: " .. ea )

      end

   end

   -- check again, as ea may have been redefined

   if type(ea) == 'table' then

      if ea.type == 'ringnspoke' then

	 local rs_pars = { x = outer.x,
			   y = outer.y,
			   rmax = outer.r - ea.opad,
			   rmin = inner.r + ea.ipad,
			   trot = ea.trot,
			   tmin = ea.tmin,
			   tmax = ea.tmax,
			   nspokes = ea.nspokes,
			   nrings = ea.nrings,
			   dist   = ea.dist }


	 for spot in rs.ringNspoke( rs_pars ) do

	    local id  = string.format( "1%03d%03d",
                                    spot.ring, spot.spoke )
	    local tag = string.format( "%s:[%03d][%03d]",
				       'rs', spot.ring, spot.spoke )

	    pinhole( tag, { x = spot.x,
			    y = spot.y,
			    z = outer.z,
			    id = tonumber(id)
			 } )
	 end

      else
	 error( "unknown entrance aperture" )
      end

   end

end

-- return a sanitized namespace
return { entrance_aperture = entrance_aperture,

	 -- expose vspec so that inputs to entrance_aperture may be sanitized
	 -- to avoid "unknown elements" errors
	 vspec = vspec
      }
