from enum import Enum
from pathlib import Path

import pytest

from ciao.tools.configuration import (Configuration as _Configuration, Parameter, NonNegative, Positive, Integer, Float,
                                      File, Normalized, Category, ListOfFiles, Boolean, Directory, ConfigurationError)


def test_basic_parameter():
    class Configuration(_Configuration):
        required = Parameter(datatype=int)
        positive = Positive(Integer(default=1000))
        non_negative = NonNegative(Float(default=0.0))

    with pytest.raises(TypeError):
        Configuration()

    conf = Configuration(required='0')
    assert conf.__parameters__['positive'] == Positive(Integer(name='positive', default=1000))
    assert conf.__parameters__['required'] == Parameter(datatype=int, name='required', default=None)
    assert 'required' in conf
    assert 'foo' not in conf

    conf = Configuration(required='0', positive=1)
    assert conf.positive == 1

    conf = Configuration(required='0', positive="1")
    assert conf.positive == 1

    conf.positive = 2
    assert conf.positive == 2

    conf.positive = "2"
    assert conf.positive == 2

    with pytest.raises(ValueError):
        conf.positive = -1

    with pytest.raises(ValueError):
        Configuration(positive='foo', required='0')

    with pytest.raises(ValueError):
        Configuration(positive='-1', required='0')

    with pytest.raises(ValueError):
        Configuration(required='0', non_negative='-1.0')

    conf = Configuration(required='0', non_negative=10.0)
    with pytest.raises(ValueError):
        conf.non_negative = -1.0


def test_file():
    class Configuration(_Configuration):
        file = File()

    with pytest.raises(TypeError):
        Configuration()

    with pytest.raises(ValueError):
        Configuration(file='/foo/bar')

    conf = Configuration(file=__file__)
    assert conf.file == Path(__file__)


def test_boolean():
    class Configuration(_Configuration):
        boolean = Boolean(default=False)

    conf = Configuration()
    assert not conf.boolean

    conf = Configuration(boolean='1')
    assert conf.boolean

    conf.boolean = '0'
    assert not conf.boolean

    conf.boolean = False
    assert not conf.boolean


def test_file_boolean():
    class Configuration(_Configuration):
        a_file = File(required=False)

    conf = Configuration(a_file='foo')
    assert conf.is_file_set(conf.a_file)

    conf = Configuration()
    assert not conf.is_file_set(conf.a_file)


def test_output_file():
    class Configuration(_Configuration):
        output_file = File(output=True)
        output_template = File(output=True, placeholders=('pippo', 'pluto'), required=False)

    with pytest.raises(ValueError) as e:
        Configuration(output_file='/foo/bar')
    assert str(e.value) == 'Path /foo/bar is in a directory that does not exist or that is not writeable'

    Configuration(output_file='/tmp/foo', output_template='pippo.pluto')  # This shouldn't fail

    with pytest.raises(ValueError) as e:
        Configuration(output_file='/tmp/foo', output_template='foo.bar')
    assert str(e.value) == 'Filename template is missing at least one of the placeholders in (\'pippo\', \'pluto\')'

    with pytest.raises(ValueError) as e:
        Configuration(output_file='/tmp/foo', output_template='/foo/bar')
    assert str(e.value) == 'Path /foo/bar is in a directory that does not exist or that is not writeable'

    Configuration(output_file='/tmp/foo', output_template='/tmp/pippo_pluto.fits')  # This shouldn't fail


def test_list_of_files(tmp_path):
    stack_file = tmp_path / 'stack.lis'
    dummies = (tmp_path / 'file1', tmp_path / 'file2')

    with stack_file.open("a") as f:
        for dummy in dummies:
            dummy.touch()
            f.write(f'{dummy}\n')

    class Configuration(_Configuration):
        file_list = ListOfFiles()

    with pytest.raises(ValueError) as e:
        Configuration(file_list=f'{str(dummies[0])},/tmp/foo')  # one files exists, the other doesn't
    assert str(e.value) == 'Path /tmp/foo does not exist or is not a file.'

    Configuration(file_list=str(dummies[0]))  # This should work

    conf = Configuration(file_list=f'@{stack_file}')
    # Not going to try all the different combinations of things you can do with the stack library.
    # We assume the stack library works, so we only check one simple use case to make sure we are calling it right.
    # I could mock the stk library, but I don't think it's worth it in this simple case.
    assert set(conf.file_list) == set(dummies)
    assert len(conf.file_list) == 2


def test_list_of_files_is_not_required(tmp_path):
    stack_file = tmp_path / 'stack.lis'

    class Configuration(_Configuration):
        file_list = ListOfFiles()

    conf = Configuration(file_list=f'@{stack_file}')
    assert len(conf.file_list) == 0


def test_list_of_files_blank(tmp_path):
    stack_file = tmp_path / 'stack.lis'
    stack_file.touch()

    class Configuration(_Configuration):
        file_list = ListOfFiles()

    conf = Configuration(file_list=f'@{stack_file}')
    assert len(conf.file_list) == 0


def test_normalized():
    class Configuration(_Configuration):
        normalized = Normalized(default=0.95)

    with pytest.raises(ValueError):
        Configuration(normalized=1.00001)

    conf = Configuration(normalized=0.5)
    with pytest.raises(ValueError):
        conf.normalized = -0.5

    class Configuration(_Configuration):
        normalized = Normalized(default=1.1)

    with pytest.raises(ValueError):
        Configuration()


def test_category():
    class Categories(Enum):
        FOO = 'foo'
        BAR = 'bar'
        BAZ = 'baz'

    class Configuration(_Configuration):
        category = Category(Categories, default=Categories.BAR)

    conf = Configuration()
    assert conf.category.value == 'bar'

    conf.category = Categories.FOO
    assert conf.category.value == 'foo'

    conf.category = 'baz'
    assert conf.category.value == 'baz'

    with pytest.raises(ValueError):
        conf.category = 'pippo'


def test_type_error_when_setting_parameter():
    class Configuration(_Configuration):
        float_par = Float()

    with pytest.raises(ValueError) as e:
        Configuration(float_par=None)

    assert str(e.value) == "Could not cast 'None' to <class 'float'> (Parameter 'float_par')"


def test_directory(tmp_path):
    (tmp_path / "file").touch()
    (tmp_path / "dir").mkdir()

    class Configuration(_Configuration):
        dir_param = Directory()

    with pytest.raises(ConfigurationError):
        Configuration(dir_param='/non_existent/path')

    with pytest.raises(ConfigurationError):
        Configuration(dir_param=tmp_path/"file")

    conf = Configuration(dir_param=tmp_path/"dir")
    assert conf.dir_param == tmp_path/"dir"

    conf = Configuration(dir_param=tmp_path/"non_existent")
    assert conf.dir_param.exists()
    assert conf.dir_param.is_dir()
