#!/opt/conda/envs/ciao-4.17.0/bin/python3
# 
#  Copyright (C) 2007-2008,2011-2012,2016,2021  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

from __future__ import print_function

from pytransform import *
from numpy import array
import numpy
from os import environ


#  Setup input/output directories
#
testin  = environ.get('TESTIN')
if testin == None :
  indir = environ.get('PWD')
else :
  indir = testin + '/test_transform'

testout = environ.get('TESTOUT')
if testout == None :
  outdir = environ.get('PWD')
else :
  outdir = testout + '/transform'


print("\n************************************************************")
print("                      WCSTransform                          ")
print("************************************************************")
#
# Wrapped and running
#   WCSTANTransform();
#   WCSTANTransform(string in_name);
#   WCSTANTransform(string in_name, list<TransParam*> in_params);
#
#   string get_name();
#   void   set_name(string in_name);
#
#   void set_parameter_list(list<TransParam*> in_params);
#   list<TransParam*> get_parameter_list();
#   void print_parameter_list();
#
#   trErrCode  set_parameter(TransParam* in_param);
#   TransParam* get_parameter(string in_name);
#   void*      get_parameter_value(string in_name);
#
#   void* apply(void* in_array, trDataType in_datatype);
#   void* invert(void* in_array, trDataType in_datatype);
#
# Ignore in Python..
#   void clear();
# 

# ----------------------------------------------------------------------  
print("\nCreate WCSTAN Transform object")
T1 = WCSTANTransform()
print("  T1 = WCSTANTransform()... OK")

print("\nDisplay Default Parameter Info")
print("  T1.print_parameter_list()...")
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nCreate WCSTAN Transform object with name.")
T2 = WCSTANTransform("Ichabod")
print("  T2 = WCSTANTransform('Ichabod')... OK")

print("\nDisplay Default Parameter Info")
print("  T2.print_parameter_list()...")
print(T2.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nName handling")

name = T2.get_name()
print("  T2.get_name()            => ", name)

T1.set_name("Hildagard")
print("  T1.set_name('Hildagard')")
print("  T1.get_name()            => ", T1.get_name())

# ----------------------------------------------------------------------  
print("\nDisplay Parameter Info")
print("  T1.print_parameter_list()...")
print(T1.print_parameter_list())

print("\n  T2.print_parameter_list()...")
print(T2.print_parameter_list())


print("\nGet Parameter List from Transform")
print("  Params = T1.get_parameter_list()")
Params = T1.get_parameter_list()
print("   Params list of type ", type(Params).__name__)
print("   Params list length ", len(Params))
print("   Params list elements of type ", type(Params[1]).__name__)
print("   Display List Elements:")
for ii in range(0, len(Params)) :
  print((Params[ii]).show())

print("\nDisplay Parameter Info Directly")
print("  T1.print_parameter_list()... ")
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nGet Specific Parameter from Transform")
print("  eqParam = T1.get_parameter('EQUINOX')")
eqParam = T1.get_parameter("EQUINOX")
print(eqParam.show())

print("\n  eqParam.set_value(1995.0)")
print("  T1.print_parameter_list()")
eqParam.set_value(1995.0)
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nGet Specific Parameter VALUE from Transform")
equinox = T1.get_parameter_value("EQUINOX")
print("  T1.get_parameter_value('EQUINOX') => ", equinox) 

equinox = T1.get_parameter_value("BLAH")
print("  T1.get_parameter_value('BLAH') => ", equinox) 

# ----------------------------------------------------------------------  
print("\nCreate a Transform using input Parameter list")
print("  - Make a Parameter list.")
crpix = TransParam("CRPIX",[4096.0,4096.0],"Reference Pixel")
print(crpix.show())
crval = TransParam("CRVAL",[123.0,456.0],"Reference Value")
print(crval.show())
cdelt = TransParam("CDELT",[1.0,1.0],"Delta per pixel")
print(cdelt.show())
Params = list([crpix, crval,cdelt])
print("  TransParam list of type ", type(Params).__name__)

print("  - Use it in constructor.")
T3 = WCSTANTransform("Rasputin", Params )
print("  T3 = WCSTANTransform('Rasputin', Params )... OK")
print("  T3.print_parameter_list()... NOTE: Only values change from default.")
print(T3.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nSet Parameter list in existing  Transform.")
print("  - Make Another Parameter list")
crpix = TransParam("CRPIX",[511.0,511.0],"Reference Pixel")
print(crpix.show())
crval = TransParam("CRVAL",[0.125,0.250],"Reference Value")
print(crval.show())
cdelt = TransParam("CDELT",[0.05,-0.05],"Delta per pixel")
print(cdelt.show())
Params = list([crpix, crval,cdelt])
print("  Parameter list of type ", type(Params).__name__)

print("  - Use it.")
print("  T2.set_parameter_list( Params )...")
T2.set_parameter_list( Params )
print(T2.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nSet/Replace an existing Parameter in the Transform.")
print("  - Create crval parameter..")
tpar = TransParam("CRVAL",[555.0,4322.0],"Reference Value")
print("  T3.set_parameter( crval )")
T3.set_parameter( tpar )
print(T3.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nModify values of an existing Parameter in the Transform.")
print("  Start Condition: T3.print_parameter_list()")
print(T1.print_parameter_list())
print("  Get parameters:  Params = T1.get_parameter_list()")
Params = T1.get_parameter_list()

print("  Modify value of CRPIX:  param.set_value([2048,2048])")
for ii in range(0, len(Params)) :
  if (Params[ii]).get_name() == "CRPIX" :
      (Params[ii]).set_value([2048,2048])
      print((Params[ii]).show())

print("  End Condition: T1.print_parameter_list()")
print("    NOTE: No 'set_parameter_list' or 'set_parameter' was done")
print("          to push the change into the Transform.             ")
print(T1.print_parameter_list())

print("  Set parameters:  Params = T1.set_parameter_list()")
Params = T1.set_parameter_list(Params)
print("  End Condition: T1.print_parameter_list()")
print(T1.print_parameter_list())

# ----------------------------------------------------------------------  
print("\nApply Transform to data")
print("- Finish setting T1 parameters") 
tpar = T1.get_parameter("CRPIX")
tpar.set_value([1.0245000000000E+03,1.0245000000000E+03])
tpar = T1.get_parameter("CRVAL")
tpar.set_value([2.0152004149394E+02,-4.2949051379390E+01])
tpar = T1.get_parameter("CDELT")
tpar.set_value([-0.000585778,0.0005857778])
tpar = T1.get_parameter("CROTA")
tpar.set_value(0.0)
tpar = T1.get_parameter("EQUINOX")
tpar.set_value(2000.0)
tpar = T1.get_parameter("EPOCH")
tpar.set_value(2000.0)

print(T1.print_parameter_list())

print("- Set input sky(X,Y) data array") 
sky = array([[0984.5, 1020.5],
             [0994.5, 1021.5],
             [1004.5, 1022.5],
             [1014.5, 1023.5],
             [1024.5, 1024.5],
             [1034.5, 1025.5],
             [1044.5, 1026.5],
             [1054.5, 1027.5],
             [1064.5, 1028.5],
             [1074.5, 1029.5]], dtype=numpy.double)

print("- Apply Transform") 
cel = T1.apply(sky)

print("  Input array length.. ", len(sky))
print("  Input array ndim.. ", sky.ndim)
print("  Input array shape.. ", sky.shape)
print("  Input array dtype.. ", sky.dtype)
print("  Output cel values are type... ", type(cel).__name__)
print("  Output array length.. ", len(cel))
print("  Output array ndim.. ", cel.ndim)
print("  Output array shape.. ", cel.shape)
print("  Output array dtype.. ", cel.dtype)

print("  X    ,   Y     ->     RA        ,     DEC ")
for ii in range(0, len(sky)) :
  print( "{:6.1f} , {:6.1f}  ->  {:.8f} , {:.8f}".format(sky[ii][0], sky[ii][1], cel[ii][0],cel[ii][1]) )


# ----------------------------------------------------------------------  
print("\nApply Inverse Transform to data")
print("- Input data array is transformed cel(RA,DEC) data")
print("- Apply Inverse Transform") 
xyout = T1.invert(cel)

print("  Input array length.. ", len(cel))
print("  Input array ndim.. ", cel.ndim)
print("  Input array shape.. ", cel.shape)
print("  Input array dtype.. ", cel.dtype)
print("  Output cel values are type... ", type(xyout).__name__)
print("  Output array length.. ", len(xyout))
print("  Output array ndim.. ", xyout.ndim)
print("  Output array shape.. ", xyout.shape)
print("  Output array dtype.. ", xyout.dtype)


print("   RA        ,     DEC       ->    X    ,   Y  ")
for ii in range(0, len(cel)) :
  print( "{:.8f} , {:.8f}  ->  {:6.1f} , {:6.1f}".format(cel[ii][0], cel[ii][1], xyout[ii][0],xyout[ii][1]) )



# ----------------------------------------------------------------------  
print("\nApply Transform to INT data")
print("- Set input sky(X,Y) data array") 
sky = array([[00, 0],
             [10, 1],
             [20, 2],
             [30, 3],
             [40, 4],
             [50, 5],
             [60, 6],
             [70, 7],
             [80, 8],
             [90, 9]], dtype=numpy.int64)

print("- Apply Transform") 
cel = T1.apply(sky)

print("  Input array length.. ", len(sky))
print("  Input array ndim.. ", sky.ndim)
print("  Input array shape.. ", sky.shape)
print("  Input array dtype.. ", sky.dtype)
print("  Output cel values are type... ", type(cel).__name__)
print("  Output array length.. ", len(cel))
print("  Output array ndim.. ", cel.ndim)
print("  Output array shape.. ", cel.shape)
print("  Output array dtype.. ", cel.dtype)

print("  X    ,   Y     ->     RA         ,     DEC ")
for ii in range(0, len(sky)) :
  print(sky[ii][0], ",", sky[ii][1], " -> ", cel[ii][0], ",", cel[ii][1])


print("\nGet Transform matrix")

myMatrix = T1.get_transform_matrix();
if myMatrix.any() :
  print("Transform Matrix: \n", myMatrix)
else :
  print("Could not extract Transform Matrix. ")
    

# ----------------------------------------------------------------------  

print("\nSet Transform Parameters from Transform Matrix")

T4 = WCSTANTransform("Buffy")

tpar = T4.get_parameter("CRVAL")
tpar.set_value([2.0152004149394E+02,-4.2949051379390E+01])
print(T4.print_parameter_list())

print("\nCall set_transform_matrix()")
T4.set_transform_matrix(myMatrix)
print(T4.print_parameter_list())

# ----------------------------------------------------------------------  

print("\nCopy T4 Transform into a new Transform called T5")

T5 = T4.copy()
T5.set_name("T5")
print(T5.print_parameter_list())

print("\nDone.\n")

