# 
#  Copyright (C) 2007-2008,2012-2013  Smithsonian Astrophysical Observatory
#
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#


set tcl_precision 12                                                                                
package provide OBSVIS 1.2

namespace eval OBSVISRaDecParser {
   namespace export New             ; # Create a new OBSVISRaDecParser
   namespace export Delete          ; # Delete the OBSVISRaDecParser
   namespace export SelfTest        ; # Test OBSVISRaDecParser
   namespace export GetRaDec        ; # Retrieve RA/Dec from a string
   namespace export GetRaDecNewFormat ; # Get last parsed RA/Dec in new format
   namespace export GetFormat       ; # Get the format of the last parsed RA/Dec
   namespace export GetErrorStatus  ; # Get the error status of the last operation
   namespace export GetErrorMessage ; # Get the error message of the last operation

   variable initFlag 0

   variable hold
   variable parseString

   variable raDec
   variable raDecFormat

   variable errStatus
   variable errMessage

   set hold ""
   set raDec(ra) [list]
   set raDec(dec) [list]
   set raDecFormat(ra) ""
   set raDecFormat(dec) ""

   set parseString ""

   set errStatus 1
   set errMessage "ERROR: Not Initialized"
}

# Initialize a new OBSVISRaDecParser
proc OBSVISRaDecParser::New {} {
   variable initFlag

   set initFlag 1

   OBSVISRaDecParser::Reset

   return
}

# Delete the current OBSVISRaDecParser
proc OBSVISRaDecParser::Delete {} {
   variable initFlag
   variable errStatus
   variable errMessage

   set initFlag 0
   OBSVISRaDecParser::Reset

   set errStatus 1
   set errMessage "ERROR: Not Initialized"

   return
}

# Retrieve whether an error was found while executing a command
# Returns 1 if an error was found 0 if not
proc OBSVISRaDecParser::GetErrorStatus {} {
   variable initFlag
   variable errStatus

   if { !$initFlag } {
      return {}
   }

   return $errStatus
}

# Retrieve the error message associated with the last error found
proc OBSVISRaDecParser::GetErrorMessage {} {
   variable errMessage
   variable initFlag

   if { !$initFlag } {
      return {}
   }

   return $errMessage
}

# Convert the ra and dec from the format of the input string to a specified
# format. By default the output format will be in degrees for both ra and
# dec. The returned value is a 2 element list, the first element being
# the ra and the second element dec.
proc OBSVISRaDecParser::GetRaDec { inputString { return_format {} } } {
   variable raDec
   variable parseString
   variable errStatus
   variable errMessage
   variable initFlag
   variable raDecFormat

   if { !$initFlag } {
      return {}
   }


   # reset all internal variables to their initial state
   OBSVISRaDecParser::Reset

   # parse the input string
   set parseString $inputString
   OBSVISRaDecParser::Parse

   # make sure there is both an ra and dec value set otherwise
   # return an error
   if { [ llength $raDec(ra)] == 0 && !$errStatus } {
      set errStatus 1
      set errMessage "ERROR: No RA information"
   } elseif { [ llength $raDec(dec) ] == 0 && !$errStatus } {
      set errStatus 1
      set errMessage "ERROR: No Dec information"
   }

   # convert the ra dec from the parsed format to the specified format
   if { !$errStatus } {
       if { $return_format == {} } { 
	   set return_format [list $raDecFormat(ra) $raDecFormat(dec)]  
       }
      set rval [OBSVISRaDecParser::ConvertFormat $return_format]
   } else {
      set rval {}
   }

   return $rval
}

# Retrieve the format found while parsing the last specified ra/dec
proc OBSVISRaDecParser::GetFormat { } {
   variable raDecFormat
   variable initFlag

   if { !$initFlag } {
      return {}
   }


   set rval [list $raDecFormat(ra) $raDecFormat(dec)]  
   return $rval
}

# Get the last specified ra/dec in a different format
proc OBSVISRaDecParser::GetRaDecNewFormat { return_format } {
  variable errStatus
  variable errMessage
  variable raDec
  variable initFlag

   if { !$initFlag } {
      return {}
   }


  # make sure there is an ra/dec previously specified
  if { [ llength $raDec(ra)] == 0 && !$errStatus } {
      set errStatus 1
      set errMessage "ERROR: No RA information"
   } elseif { [ llength $raDec(dec) ] == 0 && !$errStatus } {
      set errStatus 1
      set errMessage "ERROR: No Dec information"
   }

   # convert the ra dec from the parsed format to the specified format
   if { !$errStatus } {
      set rval [OBSVISRaDecParser::ConvertFormat $return_format]
   } else {
      set rval {}
   }

   return $rval

}

# Parse the input string. Determine its format and store the ra/dec
# in either standard format or degree format
proc OBSVISRaDecParser::Parse { } {

   variable parseString
   variable errStatus
   variable errMessage
   variable initFlag

   if { !$initFlag } {
      return
   }

   if { [string length $parseString] == 0 } {
      return
   }

   set char [ string index $parseString 0 ]

   # Remove leading spaces from the parse string
   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if [OBSVISRaDecParser::HandleRa] {
      # Remove leading spaces from the parse string

      OBSVISRaDecParser::HandleComma
      
      OBSVISRaDecParser::HandleDec
   }

}


# Parse through ra info
proc OBSVISRaDecParser::HandleRa {} {
   variable errStatus
   variable errMessage
   variable raDec
   variable raDecFormat
   variable parseString
   variable initFlag
   variable hold
   
   set rval 1

   if { !$initFlag } {
      return 0
   }


   if { 0 == [string length $parseString] } {
      return 0
   }

   if { [OBSVISRaDecParser::HandleDegreeRa] } {
      set rval 1
   } else {
      set rval [OBSVISRaDecParser::HandleHour]
   }

   if { ![OBSVISRaDecParser::HandleComma] } {
      OBSVISRaDecParser::HandlePlusMinus
   }

   # Pad Ra Accordingly
   set tmpLen [ llength [split $raDec(ra) " "]]


   if { $tmpLen == 0 } {
      set rval 0
   } elseif { $tmpLen == 1} {
      if { [string length $raDecFormat(ra)] == 0 } {
         set raDecFormat(ra) deg
      }
      set rval 1
      
      # recheck degree range as degree is the fallback if the
      # parse string was nothing else
      if { ![CheckValidRange $raDec(ra) radegree] } {
         set rval 0
         set errStatus 1
         set errMessage "ERROR: Invalid range for RA"
      }
         
   } elseif { $tmpLen == 2 } {
      if { [string length $raDecFormat(ra)] == 0 } {
         set raDecFormat(ra) standard
      }
      lappend raDec(ra) 0
      set rval 1
   } elseif { $tmpLen == 3 } {
      if { [string length $raDecFormat(ra)] == 0 } {
         set raDecFormat(ra) standard
      }
      set rval 1
   } else {
      set errStatus 1
      set errMessage "ERROR: Too many arguments. Expecting between 2 and 6"
      set rval 0
   }

   return $rval
}


proc OBSVISRaDecParser::HandleDec {} {
   variable errStatus
   variable errMessage
   variable raDec
   variable raDecFormat
   variable parseString
   variable initFlag
   variable hold
   variable parseString

   if { !$initFlag } {
      return 0
   }


   if { 0 == [string length $parseString] } {
      return 1
   }


   set rval 1
   OBSVISRaDecParser::HandlePlusMinus

   if { [OBSVISRaDecParser::HandleDegreeDec] } {
      set rval 1
   } else {
      set rval [OBSVISRaDecParser::HandleDegree]
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] > 0 } {
      lappend raDec(dec) $hold
      set hold ""
   }

   # Pad Ra Accordingly
   set tmpLen [ llength [split $raDec(dec) " "]]

   if { $tmpLen == 0 } {
      set rval 0
   } elseif { $tmpLen == 1} {
      set raDecFormat(dec) deg
      set rval 1

      # recheck degree range as degree is the fallback if the
      # parse string was nothing else
      if { ![CheckValidRange $raDec(dec) decdegree] } {
         set rval 0
         set errStatus 1
         set errMessage "ERROR: Invalid range for Dec"
      }

   } elseif { $tmpLen == 2 } {
      if { [string length $raDecFormat(dec)] == 0 } {
         set raDecFormat(dec) standard
      }
      lappend raDec(dec) 0
      set rval 1
   } elseif { $tmpLen == 3 } {
      if { [string length $raDecFormat(dec)] == 0 } {
         set raDecFormat(dec) standard
      }
      set rval 1
   } else {
      set errStatus 1
      set errMessage "ERROR: Too many arguments. Expecting between 2 and 6"
      set rval 0
   }

   if {[string length $parseString] > 0 } {
      set errStatus 1
      set errMessage "ERROR: Too many arguments. Expecting between 2 and 6"
      set rval 0

   }


   return $rval
}



# Check that the leading elements of the parse string are a number. If so
# extract those elements.
proc OBSVISRaDecParser::HandleNumeric {} {
   variable parseString
   variable hold
   variable initFlag

   if { !$initFlag } {
      return 0
   }


   set rval 1

   # the parse string is empty
   if { 0 == [string length $parseString] } {
      return 0
   }

   set char [ string index $parseString 0 ]
   set num ""

   # recursively check the parse string for a number
   set pattern "(^\[0-9]+\[.]*\[0-9]*)"
   if [regexp $pattern $parseString num] {

      # remove number from parse string
      set start [string length $num]
      set end [string length $parseString]
      set parseString [string range $parseString $start $end]

      # check for exponetial notation
      set pattern "(^\[eE]+\[-]*\[0-9]+)"
      if [regexp $pattern $parseString exp] {
         set start [string length $exp]
         set end [string length $parseString]
         set parseString [string range $parseString $start $end]

         set num [expr "$num$exp"]

      }


      # store off the number
      set num [ RemoveLeadingZeros $num ]
      set hold $hold$num

      set rval 1

   } else {
      
      set rval 0
   }

   return $rval
}

proc OBSVISRaDecParser::HandleDegreeRa {} {
   variable parseString
   variable raDecFormat
   variable raDec
   variable hold
   variable initFlag

   set rval 1

   if { !$initFlag } {
      return 0
   }

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold radegree] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]

         if {[regexp {[dD]+} $char] } {
            set parseString [OBSVISRaDecParser::ChopString $parseString ]
            set raDecFormat(ra) deg
            lappend raDec(ra) $hold
            set rval 1
            
         } elseif [OBSVISRaDecParser::HandleDegreeHourRa] {
            set rval 1
         } elseif [OBSVISRaDecParser::IsDecimal $hold] {
            set rval 1
            set raDecFormat(ra) deg
            lappend raDec(ra) $hold
         } else {
            set rval 0
         }
      }
   }

   if { $rval } {
      set hold ""
   }

   return $rval
}


proc OBSVISRaDecParser::HandleDegreeHourRa {} {
   variable parseString
   variable raDecFormat
   variable raDec
   variable initFlag
   variable hold

   set rval 1

   if { !$initFlag } {
      return 0
   }

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold hour] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]
         if {[regexp {[hH]+} $char]} {
            set parseString [OBSVISRaDecParser::ChopString $parseString ]
            set rval 1

            if [OBSVISRaDecParser::IsDecimal $hold] {
               lappend raDec(ra) [expr {$hold*15}]
               set raDecFormat(ra) hms_deg
               set hold ""
            } else {
               lappend raDec(ra) $hold
               set hold ""

               OBSVISRaDecParser::HandleColon ra

               if {![OBSVISRaDecParser::HandleMinute] } {

                   set hold_length [string length $hold]
                   if { $hold_length > 0 && \
                      ![OBSVISRaDecParser::HandleComma] && \
                      ![OBSVISRaDecParser::HandlePlusMinus] } {
                     set raDecFormat(ra) hms_deg
                     set raDec(ra) [expr {$raDec(ra)*15}]
                  } elseif { $hold_length == 0 } {
                     set raDecFormat(ra) hms_deg
                     set raDec(ra) [expr {$raDec(ra)*15}]
                  }
               }
            }
         } else {
            set rval 0
         }
      }
   }
   return $rval
}


# Check to see if the parser string is in hms format. If hours are found
# parse for minutes or hms degree format
proc OBSVISRaDecParser::HandleHour { } {
   variable parseString
   variable hold
   variable initFlag
   variable raDecFormat
   variable raDec

   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold hour] || [OBSVISRaDecParser::IsDecimal $hold] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]
         if {[regexp {[hH:0-9]+} $char] } {
            if {[regexp {[hH]+} $char] } {
               set raDecFormat(ra) hms
               set parseString [OBSVISRaDecParser::ChopString $parseString ]
            } 

            OBSVISRaDecParser::HandleColon ra

            lappend raDec(ra) $hold
            set hold ""
 
            set rval [OBSVISRaDecParser::HandleMinute]
         } else {
            set rval 0
         }
      }
   }
   return $rval

}


proc OBSVISRaDecParser::HandleMinute {} {
   variable parseString
   variable hold
   variable initFlag
   variable raDecFormat
   variable raDec
   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold minute] || [OBSVISRaDecParser::IsDecimal $hold] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]

         if {[regexp {[mM:0-9]+} $char] } {
            if {[regexp {[mM]+} $char] } {
               if { [string length $raDecFormat(ra)] == 0 } {
                  set raDecFormat(ra) hms
               }
               set parseString [OBSVISRaDecParser::ChopString $parseString ]
            }

            OBSVISRaDecParser::HandleColon ra

            lappend raDec(ra) $hold
            set hold ""

            set rval [OBSVISRaDecParser::HandleSecond]

         } else {
            set rval 0
         }
      }
   }
   return $rval
}

proc OBSVISRaDecParser::HandleSecond { } {
   variable parseString
   variable hold
   variable initFlag
   variable raDecFormat
   variable raDec
   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold second] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]

         if {[regexp {[sS0-9]+} $char] } {
            if {[regexp {[sS]+} $char] } {
               if { [string length $raDecFormat(ra)] == 0 } {
                  set raDecFormat(ra) hms
               }
               set parseString [OBSVISRaDecParser::ChopString $parseString ]
            }

            lappend raDec(ra) $hold
            set hold ""

         } else {
            set rval 0

         }
      }
   }
   return $rval
}

proc OBSVISRaDecParser::HandleDegreeDec {} {
   variable parseString
   variable raDecFormat
   variable raDec
   variable initFlag
   variable hold

   set rval 1

   if { !$initFlag } {
      return 0
   }

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 || $hold == "-" } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold decdegree] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]

         if [OBSVISRaDecParser::IsDecimal $hold] {
            set rval 1
            set raDecFormat(dec) deg
            lappend raDec(dec) $hold
         } else {
            set rval 0
         }
      }
   }

   if { $rval } {
      set hold ""
   }

   return $rval
}


proc OBSVISRaDecParser::HandleDegree { } {
   variable parseString
   variable hold
   variable initFlag
   variable raDecFormat
   variable raDec

   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 || $hold == "-" } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold decdegree] || [OBSVISRaDecParser::IsDecimal $hold] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]

         if {[regexp {[dD:0-9]+} $char] || 0 == [string length $parseString] } {
            if {[regexp {[dD]+} $char] } {
               if { [string length $raDecFormat(dec)] == 0 } {
                  set raDecFormat(dec) hms
               }
               set parseString [OBSVISRaDecParser::ChopString $parseString ]
            }
            OBSVISRaDecParser::HandleColon dec

            lappend raDec(dec) $hold
            set hold ""

            set rval [OBSVISRaDecParser::HandleArcMinute]
         } else {
            set rval 0
         }
      }
   }
   return $rval
}

proc OBSVISRaDecParser::HandleArcMinute { } {
   variable parseString
   variable hold
   variable initFlag
   variable raDecFormat
   variable raDec

   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {

      if { ![CheckValidRange $hold minute] || [OBSVISRaDecParser::IsDecimal $hold] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]

         if {[regexp {[':0-9]+} $char] || 0 == [string length $parseString] } {
            if {[regexp {[']+} $char] } {
               if { [string length $raDecFormat(dec)] == 0 } {
                  set raDecFormat(dec) hms
               }
               set parseString [OBSVISRaDecParser::ChopString $parseString ]
            }

            OBSVISRaDecParser::HandleColon dec

            lappend raDec(dec) $hold
            set hold ""

            set rval [OBSVISRaDecParser::HandleArcSecond]
         } else {
            set rval 0

         }
      }
   }
   return $rval
}

proc OBSVISRaDecParser::HandleArcSecond { } {
   variable parseString
   variable hold
   variable initFlag
   variable raDecFormat
   variable raDec

   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
   if { [string length $hold] == 0 } {
      set rval [OBSVISRaDecParser::HandleNumeric]
   }

   if { $rval } {
      if { ![CheckValidRange $hold second] } {
         set rval 0
      } else {
         set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]
         set char [ string index $parseString 0 ]
         if {[regexp {["]+} $char] || 0 == [string length $parseString] } {
            if {[regexp {["]+} $char] } {
               if { [string length $raDecFormat(dec)] == 0 } {
                  set raDecFormat(dec) hms
               }
               set parseString [OBSVISRaDecParser::ChopString $parseString ]
            }

            lappend raDec(dec) $hold
            set hold ""
         } else {
            set rval 0

         }
      }
   }
   return $rval
}

proc OBSVISRaDecParser::HandleComma { } {
   variable parseString
   variable hold
   variable errStatus
   variable errMessage
   variable initFlag
   variable raDec

   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   set char [ string index $parseString 0 ]

   if {[regexp {[,]} $char ]} {
      set parseString [OBSVISRaDecParser::ChopString $parseString ]
   
      if { [string length $hold] > 0 } {
         lappend raDec(ra) $hold
         set hold ""
      }

      set rval 1

   } else {
      set rval 0
   }

   return $rval
}


proc OBSVISRaDecParser::HandleColon { raDecIndex } {
   variable parseString
   variable hold
   variable errStatus
   variable errMessage
   variable initFlag
   variable raDec
   variable raDecFormat

   set rval 1

   if { 0 == [string length $parseString] } {
      return 0
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   set char [ string index $parseString 0 ]

   if {[regexp {[:]} $char ]} {
      set parseString [OBSVISRaDecParser::ChopString $parseString ]

      if { [string length $raDecFormat($raDecIndex)] == 0 } {
         set raDecFormat($raDecIndex) colon
      }


      set rval 1

   } else {
      set rval 0
   }

   return $rval
}


# Check to see if the leading character is a + or -.
proc OBSVISRaDecParser::HandlePlusMinus { } {
   variable parseString
   variable hold
   variable initFlag
   variable raDec

   if { !$initFlag } {
      return 0
   }

   set rval 1

   if { 0 == [string length $parseString] } {
      return $rval
   }

   set parseString [OBSVISRaDecParser::RemoveLeadingSpaces $parseString]

   set char [ string index $parseString 0 ]

   if {[regexp {[+-]+} $char ]} {
      set parseString [OBSVISRaDecParser::ChopString $parseString ]

      if { [string length $hold] > 0 } {
         lappend raDec(ra) $hold
         set hold ""
      }

      # if minus add the minus to hold buffer so negativity is not lost
      if { $char == "-" } {
         set hold $char
      }
   } else {
      set rval 0
   }

   return $rval
}




# converts degrees to standard format. the input must be in degrees
# and the type must specify whether output is in hms (hours minutes seconds)
# or arc (degrees arcminutes arcseconds) format
proc OBSVISRaDecParser::ConvertDecimal { deg type } {

   variable initFlag

   if { !$initFlag } {
      return {}
   }


   switch $type {
      hms {
         set hour [expr { int ($deg/15) }]
         set deg [expr { $deg/15.0 - $hour }]
      }
      arc {
         set hour [expr { int ($deg) }]
         set deg [expr { $deg - $hour }]
         if { $deg < 0 } {
            set deg [expr { $deg * (-1) } ]
            if { $hour == 0 } {
               set hour -$hour
            }
         }
      }
   }

   set minute [expr { int ($deg*60) }]
   set deg [expr {$deg*60 - $minute}]

   if {$type == "hms"} { 
      set second [format "%06.3f" [expr { $deg*60 }]]
      if { $second  eq "60.000" } {
         set second "00.000" 
         set minute [expr { $minute + 1}]
      } 
   } else {
      set second [format "%05.2f" [expr { $deg*60 }]]
      if { $second  eq "60.00" } {
         set second "00.00" 
         set minute [expr { $minute + 1}]
      } 
   } 
   if {$minute == 60} {
      set minute "00"
      if {$hour eq "-0"} {
         set hour "-1" 
      } else { 
         if {$hour < 0} {
             set hour [expr { $hour -1}]
         } else {
             set hour [expr { $hour +1}]
         }
      }
   }
   set minute [format "%02d" $minute]
   if { $hour eq "-0" } {
      set hour "-00"
   } else {
      if { $hour < 0 } {
         set hour [format "%03d" $hour]
      } else { 
         set hour [format "%02d" $hour]
      } 
   } 
   if {$type == "hms"} {
      if { $hour eq "24" } {
         set hour "00"
      } 
   } 
   return [list $hour $minute $second]
}

# converts standard coordinates to degrees. the coords must have three
# numbers. their format is specified by type. type must be either hms
# (hours minutes seconds) or arc (degrees arcminutes arcseconds).
proc OBSVISRaDecParser::ConvertStandard { coords type } {
   variable errStatus 
   variable errMessage
   variable initFlag

   if { !$initFlag } {
      return ""
   }


   set deg 0

   # make sure there are three coordinate to convert
   if { [llength $coords] != 3 } {
      set errStatus 1
      set errMessage "ERROR: Invalid coordinates values"
      return
   }

   set deg [expr {[lindex $coords 2]/60.0}]
   set deg [expr {($deg + [lindex $coords 1])/60.0}]
   switch $type {
      hms {
         set deg [expr {($deg + [lindex $coords 0])*15.0}]
      }
      arc {

         if { [ regexp {[-].*} [lindex $coords 0] ] } {
            set deg [expr {[lindex $coords 0] - $deg}]
         } else {
            set deg [expr {$deg + [lindex $coords 0]}]
         } 
      }
   }


   return $deg

}


# Convert the already parse ra/dec arrays into a new format. Internally
# ra/dec arrays are stored either as decimal degrees (a single value) or
# standard (h m s d arcmin arcsec). However, the formats that they were
# actually written is are stored off. hms, colon and standard are all
# stored internally as standard format. deg and deg_hms are stored in
# degree formats. Return a two element list with elements of the ra/dec
# array in the specified format
proc OBSVISRaDecParser::ConvertFormat { new_format } {
   variable raDec
   variable raDecFormat
   variable errStatus
   variable errMessage
   variable initFlag

   if { !$initFlag } {
      return {}
   }


   set index ra
   set rval [list]

   # go through both ra and dec and convert formats
   for { set ii 0 } { $ii < 2 } { incr ii } {

      # check which format to convert to
      switch [lindex $new_format $ii] {
         deg {

            # if the current format is degrees or hms_deg just capture
            # the current value. 
            if { $raDecFormat($index) == "deg" || \
                 $raDecFormat($index) == "hms_deg" } {
               lappend rval $raDec($index)
            } else {
               # otherwise convert the standard format to degrees
               if { $index == "ra" } {
                  lappend rval [ConvertStandard $raDec($index) hms]
               } else {
                  lappend rval [ConvertStandard $raDec($index) arc]
               } 
            }
         }
         hms_deg {

            # the dec component cannot be represented in hms_deg format
            if { $index == "dec" } {
               set errStatus 1
               set errMessage "ERROR: Cannot convert dec to hms_deg format"
               set rval {}
               break

            } else {
 
               # if the current format is degrees or hms_deg just capture
               # the current value and divide by 15 to convert to hours.
               if { $raDecFormat($index) == "deg" || \
                    $raDecFormat($index) == "hms_deg" } {
                  lappend rval [expr {$raDec($index) / 15}]h
               } else {
                  # otherwise convert the standard format to degrees
                  # and divide by 15 to convert to hours.
                  lappend rval [expr { [ConvertStandard $raDec($index) hms] / 15}]h
               }
            }
         }
         hms {
            # if the current format is degrees or hms_deg convert the
            # degree value to standar format
            if { $raDecFormat($index) == "deg" ||
                 $raDecFormat($index) == "hms_deg" } {
               if { $index == "ra" } {
                  set tmpRaDec [ConvertDecimal $raDec($index) hms]
               } else {
                  set tmpRaDec [ConvertDecimal $raDec($index) arc]
               }
            } else {
               # otherwise just capture the value
               set tmpRaDec $raDec($index)
            }

            # parse each element of the standard format and add the
            # appropriate symbols
            set tmpList {}
            if { $index == "ra" } {
               lappend tmpList [lindex $tmpRaDec 0]h
               lappend tmpList [lindex $tmpRaDec 1]m
               lappend tmpList [lindex $tmpRaDec 2]s
            } else {
               lappend tmpList [lindex $tmpRaDec 0]d
               lappend tmpList [lindex $tmpRaDec 1]'
               lappend tmpList [lindex $tmpRaDec 2]\"

               # list automatically install the backslash. remove it
               # from the list
               set tmpList [regsub {\\} $tmpList {}]
            }
            lappend rval $tmpList
         }
         colon {
            # if the current format is degrees or hms_deg convert the
            # degree value to standar format
            if { $raDecFormat($index) == "deg" ||
                 $raDecFormat($index) == "hms_deg" } {
               if { $index ==  "ra" } {
                  set tmpRaDec [ConvertDecimal $raDec($index) hms]
               } else {
                  set tmpRaDec [ConvertDecimal $raDec($index) arc]
               }
            } else {
               # otherwise just capture the value
               set tmpRaDec $raDec($index)
            }

            # add colons between each of the standard elements
            set tmpList "[lindex $tmpRaDec 0]:[lindex $tmpRaDec 1]:[lindex $tmpRaDec 2]"

            lappend rval $tmpList
         }
         standard {
            # if the current format is degrees or hms_deg convert the
            # degree value to standar format
            if { $raDecFormat($index) == "deg" || 
                 $raDecFormat($index) == "hms_deg" } {
               if { $index == "ra" } {
                  lappend rval [ConvertDecimal $raDec($index) hms]
               } else {
                  lappend rval [ConvertDecimal $raDec($index) arc]
               }
            } else {
               # otherwise just capture the value
               lappend rval $raDec($index)
            }
         }
      }
      set index dec
   }

   return $rval
         
}

# Chop the front of the string. str is the string to be chopped 
# and start is the index which should start return string
proc OBSVISRaDecParser::ChopString { str { start 1 } } {
   variable initFlag

   if { !$initFlag } {
      return {}
   }

  return [string range $str $start [string length $str]]
}

# Remove leading spaces from the input string
proc OBSVISRaDecParser::RemoveLeadingSpaces { str } {
   variable parseString
   variable initFlag

   if { !$initFlag } {
      return {}
   }


   if { 0 == [string length $str] } {
      return $str
   }

   set str [string trimleft $str { }]


   return $str
}

# Remove leading zeros from input string. This function is necessary
# so that tcl doesn't get confused with octal format
proc OBSVISRaDecParser::RemoveLeadingZeros { str } {
   variable parseString
   variable initFlag

   if { !$initFlag } {
      return {}
   }


   if { 0 == [string length $str] } {
      return $str
   }

   set str [string trimleft $str 0]

   # add a zero if the string is emtpy or the leading character is
   # is a decimal point.
   if { [string length $str] == 0 || [string index $str 0] == "." } {
      set str 0$str
   }

   return $str
}

proc OBSVISRaDecParser::IsDecimal { val } {
   variable initFlag

   if { !$initFlag } {
      return {}
   }

   return [regexp {[.]+} $val]
}

proc OBSVISRaDecParser::CheckValidRange { val type } {

   variable initFlag

   if { !$initFlag } {
      return {}
   }

   set rval 0

   switch $type {
      radegree {
         set rval [expr {$val >= 0 && $val <360}]
      }
      decdegree {
         set rval [expr {$val >= -90 && $val <= 90}]
      }
      hour {
         set rval [expr {$val >= 0 && $val <24}]
      }
      minute {
         set rval [expr {$val >= 0 && $val <60}]
      }
      second {
         set rval [expr {$val >= 0 && $val <60}]
      }
   }

   return $rval
}

# Re-initialize all variables
proc OBSVISRaDecParser::Reset {} {
   variable raDec
   variable hold
   variable parseString
   variable errStatus
   variable errMessage
   variable raDecFormat
   variable initFlag

   if { !$initFlag } {
      return
   }


   set hold ""
   set raDec(ra) [list]
   set raDec(dec) [list]
   set raDecFormat(ra) ""
   set raDecFormat(dec) ""

   set parseString ""

   set errStatus 0
   set errMessage ""

}

# Clear the error variables
proc OBSVISRaDecParser::ResetErr {} {
   variable errStatus
   variable errMessage
   variable initFlag

   if { !$initFlag } {
      return
   }


   set errStatus 0
   set errMessage ""

}

# Test the OBSVISRaDecParser routines
proc OBSVISRaDecParser::SelfTest {} {

   variable initFlag

   set totalTests 0
   set passedTests 0

   OBSVISRaDecParser::New

   incr totalTests

   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::New..."

   if { !$initFlag } {
      puts "FAILED"
   } else {
      puts "PASSED"
      incr passedTests
   }

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec standard format..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {23 59 59.9999 -89 59 59.999} {standard standard} \
         {{23 59 59.9999} {-89 59 59.999}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetFormat standard..."
   OBSVISRaDecParser::CheckTest GetFormat \
         {} {} {standard standard} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec colon format..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {23:59:59.9999 -89:59:59.999} {standard standard} \
         {{23 59 59.9999} {-89 59 59.999}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetFormat colon..."
   OBSVISRaDecParser::CheckTest GetFormat \
         {} {} {colon colon} {}
   
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec degree format..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {23.7654, +0.6857} {standard standard} \
         {{01 35 03.696} {00 41 08.52}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetFormat deg..."
   OBSVISRaDecParser::CheckTest GetFormat \
         {} {} {deg deg} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec hms format..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {09h 16m 54.28s 32d 15' 6.1"} {standard standard} \
         {{9 16 54.28} {32 15 6.1}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetFormat hms..."
   OBSVISRaDecParser::CheckTest GetFormat \
         {} {} {hms hms} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec degree format with hour..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {10.9876h, +14} {standard standard} \
         {{10 59 15.360} {14 00 00.00}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetFormat degree format with hour..."
   OBSVISRaDecParser::CheckTest GetFormat \
         {} {} {hms_deg deg} {}


   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec comma..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {14 12, 16 10} {standard standard} \
         {{14 12 0} {16 10 0}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec plus..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {14 12 +16 10} {standard standard} \
         {{14 12 0} {16 10 0}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec dec..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {14 12 16d 10} {standard standard} \
         {{14 12 0} {16 10 0}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec ambiguous.."
   OBSVISRaDecParser::CheckTest  GetRaDec \
         {14 12 16 10} {standard standard} \
         {{14 12 16} {10 00 00.00}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec spaces..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {  09  h  16  m 54.28  s  32   d   15 ' 6.1  "  } {standard standard} \
         {{9 16 54.28} {32 15 6.1}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec spaces 2..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {09  :  16   54.28  ,  32       15 ' 6.1  "  } {standard standard} \
         {{9 16 54.28} {32 15 6.1}} {}


   incr totalTests 1   
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec short colon ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {1:2, 4} {} \
        {1:2:0 4} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec exp test ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {1.0e2, -4e-5} {} \
        {100.0 -4e-05} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec decimal test ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {0.0e2, 0 0 0.1} {} \
        {0.0 {0 0 0.1}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec negative 0 test ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {00 00 00 -00 05 00} {deg deg} \
        {0.0 -0.0833333333333} {}

      incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec negative 0 test2 ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {359.0 -0.05} {standard standard} \
        {{23 56 00.000} {-00 03 00.00}} {}


   # Mix and Match
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec  hms/colon..."

   OBSVISRaDecParser::CheckTest GetRaDec \
         {09h 16m 54.28s -89:59:59.999} {} \
         {{9h 16m 54.28s} -89:59:59.999} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetFormat hms/colon..."
   OBSVISRaDecParser::CheckTest GetFormat \
         {} {} {hms colon} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec standard/deg..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {09 16 54.28, -89.9999997222} {colon hms} \
         {9:16:54.28 {-90d 00' 00.00"}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec hms_deg/standard..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {10.9876h, +14} {hms_deg standard} \
         {10.9876h {14 00 00.00}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec small degrees..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {0 0 0 0 20 0} {deg deg} \
        {0.0 0.333333333333} {}



   # Get Convert Formats
   OBSVISRaDecParser::GetRaDec {23 59 59.9999 -89 59 59.999}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDecNewFormat standard..."
   OBSVISRaDecParser::CheckTest GetRaDecNewFormat \
        {standard standard} {} \
        {{23 59 59.9999} {-89 59 59.999}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDecNewFormat hms..."
   OBSVISRaDecParser::CheckTest GetRaDecNewFormat \
        {hms hms} {} \
        {{23h 59m 59.9999s} {-89d 59' 59.999"}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDecNewFormat deg..."
   OBSVISRaDecParser::CheckTest GetRaDecNewFormat \
        {deg deg} {} \
        {359.999999583 -89.9999997222} {}
        
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDecNewFormat colon..."
   OBSVISRaDecParser::CheckTest GetRaDecNewFormat \
        {colon colon} {} \
        {23:59:59.9999 -89:59:59.999} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDecNewFormat colon deg..."
   OBSVISRaDecParser::CheckTest GetRaDecNewFormat \
        {colon deg} {} \
        {23:59:59.9999 -89.9999997222} {}

   
   # Bad values
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - no data..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {} {standard standard} {} "ERROR: No RA information"
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - one ra..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {9} {standard standard} {} "ERROR: No RA information"
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - two ra..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {9 1} {standard standard} {} "ERROR: No Dec information"
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - no dec..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {9 1 2} {standard standard} {} "ERROR: No Dec information"
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - too much data..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {9 1 2 3 4 5 6} {standard standard} {} "ERROR: Too many arguments. Expecting between 2 and 6"

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - convert dec to hms_deg..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {10.9876h, +14} {hms_deg hms_deg} {} "ERROR: Cannot convert dec to hms_deg format"

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - expecting a numerical value ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {1h m} {} {} "ERROR: Too many arguments. Expecting between 2 and 6"

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - expecting a numerical value ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {h} {} {} "ERROR: No RA information"

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - bad ra degree ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {360, 3} {} {} "ERROR: Invalid range for RA"

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - bad dec degree ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {3, 360} {} {} "ERROR: Invalid range for Dec"

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec Error - bad format ..."
   OBSVISRaDecParser::CheckTest GetRaDec \
        {10 -1, 5} {} {} "ERROR: Too many arguments. Expecting between 2 and 6"

  #WLM - wrapping roundoff/precision
   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec format - no rounding..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {139.249995833 -0.999997222} {colon colon} \
         {09:16:59.999 -00:59:59.99} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec format - rounding..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {139.249999958 -0.999999972} {colon colon} \
         {09:17:00.000 -01:00:00.00} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec format - 24 hour..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {359.999999583 -23.9999999722} {hms hms} \
         {{00h 00m 00.000s} {-24d 00' 00.00"}} {}

   incr totalTests 1
   puts -nonewline "$totalTests. Testing OBSVISRaDecParser::GetRaDec format - 23 hour..."
   OBSVISRaDecParser::CheckTest GetRaDec \
         {359.9999899 -23.99999722} {hms hms} \
         {{23h 59m 59.998s} {-23d 59' 59.99"}} {}







   OBSVISRaDecParser::Delete

   puts "SUMMARY $passedTests out of $totalTests tests passed"

   return [list $totalTests $passedTests]
  

}

proc OBSVISRaDecParser::CheckTest {func input input2 output outputErrMsg} {
   variable errStatus
   variable errMessage

   upvar passedTests passedTestCnt

   if { $func == "GetRaDec" } {
      if { [string length $input2] > 0 } {
         set out [OBSVISRaDecParser::GetRaDec $input $input2]
      } else {
         set out [OBSVISRaDecParser::GetRaDec $input]
      }
   } elseif { $func == "GetFormat" } {
      set out [OBSVISRaDecParser::GetFormat]
   } elseif { $func == "GetRaDecNewFormat" } {
      set out [OBSVISRaDecParser::GetRaDecNewFormat $input]
   }
   if { $out == $output } {
      if { [string length $outputErrMsg] > 0 } {
         if { $errStatus && $outputErrMsg == $errMessage } {
            puts "PASSED"
            incr passedTestCnt
         } else {
            puts "FAILED"
         }
      } else {
         puts "PASSED"
         incr passedTestCnt
      }
   } else {
       puts "FAILED"
   }
}



