
# --8<--8<--8<--8<--
#
# Copyright (C) 2006-2011 Smithsonian Astrophysical Observatory
#
# This file is part of MST-Raytrace
#
# MST-Raytrace is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# -->8-->8-->8-->8--

package MST::Raytrace;

use strict;
use warnings;

use base 'Exporter';

our @EXPORT_OK =
  qw(
     write_summary
     read_summary
     write_tot_wt
     read_tot_wt
     get_tot_wt
     get_focus
     read_focus
     get_enen
     get_centroid
     read_energy_mlcfg
     read_energy_rdb
     ENEN_R
     ENEN_F
    );

use RDB;
use IO::File;
use Carp;
use YAML::XS;

use Params::Validate;
use String::Interpolate::RE qw( strinterp );

our $VERSION = '1.5.7';

sub write_tot_wt {

    my ( $file, $tot_wt ) = @_;

    my $fh = IO::File->new( $file, 'w' )
      or die( "unable to create $file\n" );

    ## no critic (ProhibitAccessOfPrivateData)

    $fh->printf( "%s:\t%f\n", $_, $tot_wt->{$_} )
      for keys %$tot_wt;
}

sub read_tot_wt
{
  my $file = shift;
  my %field;

  my $fh = IO::File->new($file) or
    croak( __PACKAGE__, "::get_tot_wt: unable to open $file\n" );

  while ( my $line = <$fh>)
  {
    next unless
      $line =~ /^(n|wt|n_ghosts|wt_ghosts):\s*((?:[+-]?(?:\d+[.]?\d*|[.]\d+)(?:[dDeE][+-]?\d+)?))\s*/;
    $field{$1} = $2;
  }
  croak( __PACKAGE__,
	 "::get_tot_wt: error parsing tot_wt output file $file\n" )
    unless exists $field{n} && exists $field{wt};

  return wantarray ? @field{ 'n', 'wt', 'n_ghosts', 'wt_ghosts' } : \%field;
}

=pod

=for pod_coverage

=head3 get_tot_wt

=cut

*get_tot_wt = \&read_tot_wt;

sub read_focus
{
  my $file = shift;
  my $zfocus;
  my @stuff;

  my $fh = IO::File->new( $file )
    or croak( __PACKAGE__, "::read_focus: unable to open focus file `$file'\n");

  while (<$fh>)
  {
    if (/GENERAL SYSTEM FOCUS .*= (.*)$/)
    {
      ($zfocus = $1) =~ s/D/E/;
    }
    s/D/E/g;
    @stuff = split(' ');
  }

  croak( __PACKAGE__, "::read_focus: error parsing focus file `$file'\n" )
    unless defined $zfocus && 4 != @stuff;

  $stuff[3] += $zfocus;

  return wantarray ? @stuff[1,2,3] 
                   : { x => $stuff[1], y => $stuff[2], z => $stuff[3] };

}

=pod

=for pod_coverage

=head3 get_focus

=cut

*get_focus = \&read_focus;

sub write_summary {

    my ( $tag, @summary ) = @_;

    my $file = "$tag.yml";

    ## no critic (ProhibitAccessOfPrivateData)
    # generate hash keyed off of shell id
    my %summary = map { $_->{shell} => $_ } @summary;

    YAML::XS::DumpFile( $file, \%summary );
}

sub read_summary {

    my ( $tag ) = @_;

    my $file = "$tag.yml";

    return YAML::XS::LoadFile( $file );
}


use constant ENEN_R => 1;
use constant ENEN_F => 2;

sub get_enen
{
  my ( $file, $type, $attr ) = @_;

  $attr = {} unless defined $attr;

  croak( __PACKAGE__, "::get_enen: unknown enen type\n" )
    unless ENEN_R == $type || ENEN_F == $type;

  my $rdb = RDB->new($file) or
    croak( __PACKAGE__, "::get_enen: error opening/parsing $file\n" );

  my %data;
  my %results = ();
  while ( $rdb->read( \%data ) )
  {
    ## no critic (ProhibitAccessOfPrivateData)
    if ( ENEN_R == $type )
    {
      my $key = $attr->{FloatKeys} ? $data{radius}+0 : $data{radius};
      $results{$key} = { weight => $data{weight},
				n => $data{n},
				fraction => $data{fraction}
			      };
    }
    else
    {
      my $key = $attr->{FloatKeys} ? $data{fraction}+0 : $data{fraction};
      $results{$key} = { weight => $data{weight},
				n => $data{n},
				radius => $data{radius}
			      };
    }
  }

  return \%results;
}

sub get_centroid
{
  my $file = shift;

  my $rdb = RDB->new( $file )
    or croak( __PACKAGE__, "::get_centroid: unable to open $file\n" );

  my %data;
  $rdb->read(\%data)
    or croak ( __PACKAGE__, "::get_centroid: error reading $file\n" );

  %data;
}


####################################

sub read_energy_mlcfg
{

    my %arg = validate( @_, { file   => 1,
			      layer  => 1,
			      emin   => { default => undef },
			      emax   => { default => undef },
			      shell  => { default => 1 },
                              geo    => { default => 'p' },
			    } );

    my $epfx = __PACKAGE__ . "::read_energy_mlcfg";

    $arg{file} .= '.cnf' unless $arg{file} =~ /.cnf$/;

    require Config::Wild;
    require RDB;

    my $cfg = Config::Wild->new($arg{file});

    $cfg->set( shell => $arg{shell} );
    $cfg->set( geo   => $arg{geo} );
    my $optic = $arg{geo} . $arg{shell};

    # need to get list of energies; any old shell will do
    my $ml_db = $cfg->value( "reflect_${optic}_db" ) or
      die( "$epfx: no multi-layer db in $arg{file}?\n" );


    # also need to get post_en_corr file if it exists, to get the intersection
    # of the energy ranges
    if ( my $post_en_corr_file = $cfg->value( 'post_en_corr_file' ) )
    {
	my $mincol = $cfg->value('post_en_corr_begin');
	my $maxcol = $cfg->value('post_en_corr_end');

	my $rdb = RDB->new( $post_en_corr_file )
	  or die( "$epfx: error opening $post_en_corr_file\n" );

	my %data;
	$rdb->read(\%data);
	my ( $lemin, $lemax ) = @data{$mincol, $maxcol};

	while ( $rdb->read(\%data ) )
	{
	    $lemin = $data{$mincol} if $lemin > $data{$mincol};
	    $lemax = $data{$maxcol} if $lemax < $data{$maxcol};
	}

	# narrow energy bounds
	$arg{emin} = $lemin unless defined $arg{emin};
	$arg{emin} = $lemin if $arg{emin} < $lemin;

	$arg{emax} = $lemax unless defined $arg{emax};
	$arg{emax} = $lemax if $arg{emax} > $lemax;
    }


    my $rdb = new RDB $ml_db
      or die( "$epfx: error opening $ml_db\n" );

    # grab specified layer entry
    my %data;
    my $nrow = 1;
    my $found = 0;
    while ( ! $found && $rdb->read(\%data) )
    {
	$found = $data{material} eq $arg{layer} || $nrow++ eq $arg{layer};
    }
    $rdb->close();


    die( "$epfx: can't find layer $arg{layer} in $ml_db\n" )
      unless $found;

    # the optconst RDB file name may have interpolated environmental variables
    read_energy_rdb( file => strinterp($data{optconst_rdb}),
		     ecol => 'energy',
		     emin => $arg{emin},
		     emax => $arg{emax} );
}

sub read_energy_rdb
{
    my %arg = validate( @_,
		      {
		       file => 1,
		       ecol => { default => 'energy' },
		       emin => { default => undef },
		       emax => { default => undef }
		      });

    require RDB;

    my $epfx = __PACKAGE__ . "::read_energy_rdb";

    my $rdb = RDB->new( $arg{file} )
      or die( "$epfx: error opening $arg{file}\n" );


    die( "$epfx: $arg{file}: nonexistent column '$arg{ecol}'\n" )
      unless $rdb->type( $arg{ecol} );

    my $energy;
    my @energies;
    eval { $rdb->bind( { $arg{ecol} => \$energy } ) };
    die( "$arg{file}:", $@ ) if $@;

    while ( $rdb->read )
    {
	next if defined $arg{emin} && $energy < $arg{emin};
	next if defined $arg{emax} && $energy >= $arg{emax};

	push @energies, $energy;
    }

    @energies;
}


1;


__END__

=head1 NAME

MST::Raytrace - various raytrace related functions

=head1 SYNOPSIS

  use MST::Raytrace qw( get_focus get_tot_wt get_enen
                        get_centroid ENEN_R ENEN_F
			read_energy_mlcfg read_energy_rdb
		      );

  ( $x, $y, $z ) = get_focus( $file );
  ( $cnt, $wt, $cnt_ghosts, $wt_ghosts ) = get_tot_wt( $file );
  $results = get_enen( $file, $type );
  %centroid_results = get_centroid( $file );

  @energies = read_energy_mlcfg( %args );
  @energies = read_energy_rdb( %args );

=head1 DESCRIPTION

The B<MST::Raytrace> module provides a galimaufry of routines for
dealing with some raytrace issues.  Generally, the routines will
C<die> with an appopriate error message upon error; the caller should
use C<eval> to catch the errors.

No functions or constants are exported by default.

=head1 FUNCTIONS

=over 8

=item read_tot_wt

  ( $n, $wt, $n_ghosts, $wt_ghosts ) = read_tot_wt( $file );
  $hash = read_tot_wt( $file)

B<read_tot_wt> parses a logfile generated by B<tot_wt> and returns the
count and weight of the rays.  C<$n> and C<$wt> are the total count
and weight including ghost rays.  C<$n_ghosts>, C<$wt_ghosts> are
the total count and weight for the ghost rays by themselves.  (The
number and count of non-ghost rays are C<$n - $n_ghosts>, and
C<$wt - $wt_ghosts>, respectively.)

If called in scalar context, it returns a hash references to the data,
with keys C<n>, C<wt>, C<n_ghosts>, C<wt_ghosts>.

=item write_tot_wt

  write_tot_wt( $file, \%data );

B<write_tot_wt> creates a B<tot_wt> compatible file from the
input data.  It writes out whatever fields are in the hash.
It does not check what they are, but they should be:

  n wt n_ghosts wt_ghosts

=item read_focus

  ( $x, $y, $z ) = read_focus( $file )
  $hash = read_focus( $file )

B<read_focus> parses an output file create by the C<OSAC> focus routine and
sums the "GENERAL SYSTEM FOCUS" and the "GLOBAL OPTIMAL FOCUS" Z-plane
coordinate.  It assumes that ZOFF is wrt STD (as set in the .gi file).

In list mode it returns the individual components; in scalar mode
it returns a hash with keys C<x>, C<y>, and C<z>.

=item write_sumary

   write_summary( $base, @summaries )

Write one or more raytrace summaries as separate records to an output
file. The file name is composed of C<$base> and a suffix (currently
C<.yml>) which may change depending upon the format of raytrace summary
files.  Each summary is a hash; the only mandatory element is C<shell>,
which is the shell id.

Possible elements are:

=over

=item shell

The shell number. This is required.

=item tot_wt_in

The value should be a hash containing elements extracted from a
B<tot_wt> logfile.  This is optional.

=item tot_wt_out

The value should be a hash containing elements extracted from a
B<tot_wt> logfile.  This is optional.

=item focus

The value should be a hash containing elements extracted from a
B<saofocus> logfile.  This is optional.

=item param

The value should be a hash containing the original parameters
for the raytrace.

=back

=item read_summary

  $hash = read_summary( $base )

Read raytrace summaries from a file.  The file name is composed of
C<$base> and a suffix (currently C<.yml>) which may change depending
upon the format of raytrace summary files.  The returned hash
is keyed off of shell ids.

=item write_summary

  write_summary( $base, $hash )

Write raytrace summaries from a file.  The file name is composed of
C<$base> and a suffix (currently C<.yml>) which may change depending
upon the format of raytrace summary files.  The passed hash
must be keyed off of shell ids.

=item get_enen

  $results = get_enen( $file, $type )
  $results = get_enen( $file, $type, \%attr )

B<get_enen> parses an output file created by the C<enen-p> program, It
returns a reference to a hash keyed off of either the radius (if
I<$type> is C<ENEN_R>) or the fraction (if I<$type> is C<ENEN_F>).
The elements of the hash are references to hashes containing the rest
of the data, with the keys C<weight>, C<n>, and either C<fraction> (if
I<$type> is C<ENEN_R> ) or C<radius> ( if I<$type> is C<ENEN_F> ).


C<%attr> is an optional hash of attributes.  The following are recognized:

=over

=item FloatKeys

If true, the keys are stored asq floating point numbers.  The default is that they are stored as strings.

=back

Here's what it looks like:

  use Data::Dumper;
  $results = get_enen( $file, ENEN_F );
  print Dumper $results;

might result in

  $VAR1 = {
          '0.70000' => {
                         'n' => '26',
                         'weight' => '25.20000',
                         'radius' => '0.0372688'
                       },
          '0.40000' => {
                         'n' => '15',
                         'weight' => '14.40000',
                         'radius' => '0.0205026'
                       },
          '0.10000' => {
                         'n' => '4',
                         'weight' => '3.60000',
                         'radius' => '0.010121'
                       }
          };

By default the keys to the hash are strings, so if you try

  print Dumper $results->{0.7};

you'll get

  $VAR1 = undef;

To have them be floating point numbers, set the B<FloatKeys> attribute:

  $results = get_enen( $file, ENEN_F, { FloatKeys => 1 } );
  print Dumper $results->{0.7};

results in

$VAR1 = {
          'n' => '26',
          'weight' => '25.20000',
          'radius' => '0.0372688'
        };


Another example:

  $results = get_enen( $file, ENEN_R );
  foreach my $radius ( keys %$results )
  {
    print $results{$radius}{weight}, ' ', $results{$radius}{fraction};
  }



Upon error, it C<croak>'s; the calling routine should use C<eval> to catch
the error.

=item get_centroid

  %centroid_results = get_centroid( $file );

This parses the table output by the centroid routine.  It returns
a hash with the following keys:
C<n>,
C<x_ave>,
C<y_ave>,
C<z_ave>,
C<x_dev>,
C<y_dev>,
C<z_dev>,
C<x_rms>,
C<y_rms>,
C<z_rms>, and
C<r_rms>.

=item read_energy_mlcfg

  @energies = read_energy_mlcfg( %args );

Return a list of energies based upon a B<trace-nest> configuration
file.  It uses the energies from the optical constant data for a given
surface layer (from the multilayer configuration file) and limits the
energy range to be compatible with the post-collimator correction (if
present).

The arguments are:

=over

=item B<file>

The name of the B<trace-nest> configuration file.  The actual file
should end in a suffix of C<.cnf>.  The suffix will be added if necessary.
This is required.

=item B<layer>

This specifies which layer to use.  It is either the name of the
material or the layer number (where C<1> is the top layer).
This is required.

=item B<emin>

Specify a lower bound for the energies. Energies less than this value
are not returned. If not specified there is no minimum cutoff.

=item B<emax>

Specify an upper bound for the energies.  Energies greater than or
equal to this value are not returned. If not specified there is no
maximum cutoff.

=item B<shell>

The shell id to use to resolve the multi-layer databse name in the
=trace-nest= configuration file.  It defaults to C<1>.

=item B<shell>

The geometry of the shell to use to resolve the multi-layer databse
name in the =trace-nest= configuration file.  It defaults to C<p>.

=back


=item read_energy_rdb

  @energies = read_energy_rdb( %args );

Read a list of energies from an RDB file, optionally imposing minimum
or maximum values.

The arguments are:

=over

=item B<file>

The name of the RDB file.  This is required.

=item B<ecol>

The name of the column to read.  It defaults to C<energy>.

=item B<emin>

Specify a lower bound for the energies. Energies less than this value
are not returned. If not specified there is no minimum cutoff.

=item B<emax>

Specify an upper bound for the energies.  Energies greater than or
equal to this value are not returned. If not specified there is no
maximum cutoff.

=back


=back

=head1 Author

Diab Jerius ( djerius@cfa.harvard.edu )
