-- --8<--8<--8<--8<--
--
-- Copyright (C) 2011 Smithsonian Astrophysical Observatory
--
-- This file is part of saotrace.baffles
--
-- saotrace.baffles is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or (at
-- your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.
--
-- -->8-->8-->8-->8--

local seq = require('pl.seq')
local assert = assert
local string = require('string')
local error = error
local geometry = require( 'saotrace.geometry' )
local rdb = require('rdb')
local tables = require( 'saotrace.suplib.tables' )
local log = require('saotrace.log')

local vobj = require( 'validate.args' ):new()
vobj:setopts{ named = true}

local modname = select(1,...)
local M = {}
if setfenv then
   setfenv( 1, M ) -- 5.1
else
   __ENV = M       -- >= 5.2
end

vspec = {

   shell  = { type = 'posnum' },
   optic  = { type = 'string' }, -- e.g. p, s for primary and secondary
   cols   = { vtable = {
		 shell = { type = 'string', default = 'shell' },
		 optic = { type = 'string', default = 'optic' },
		 rthick = { type = 'string', default = 'rthick' },
		 x = { type = 'string' },
		 y = { type = 'string' },
		 z = { type = 'string' },
		 r = { type = 'string' },
	      },
	   },
   db     = { type = 'string' },
   inner_disk = {
      optional = true,
      default_is_nil = true,
      vtable = { x = { type = 'number', default = 0 },
		 y = { type = 'number', default = 0 },
		 z = { type = 'number' },
		 r = { type = 'posnum' },
	      },
   },
}


--- create an ideal baffle from the geometry database
function create( ... )

   -- can't create logger at module load time, as it needs to
   -- be initialized by some other code first.

   local logger = log.logger()

   local ok, args = vobj:validate( vspec, ... )
   assert( ok, args )

   -- select the previous, current, or next shells
   local shell = args.shell
   local optic = args.optic


   logger:info( "%s: Reading geometry information for mirror '%s%d' from '%s'", modname, optic, shell, args.db )

   local iter, vars = geometry.read{ file = args.db,
				     mode = 'iterator' }

   -- rename the fields to what we want them to be
   local extract = function(x) return tables.extract(x, nil, args.cols) end

   -- only grab those for the specified optic type
   local filter = function(x) return x.optic == optic end

   -- sort them in reverse radii
   local sort   = function(e1, e2) return e1.r > e2.r end

   -- the full iterator
   local entries = seq(iter):map(extract):filter( filter ):sort( sort )

   local inner, match, outer, prev

   -- find shell and next smaller shell to create a circular aperture
   -- if it's the innermost shell, use it's next largest neighbor
   for current in entries do

      if current.shell == shell then

	 match = current
	 outer = prev

      elseif prev and prev.shell == shell then

	 inner = current
	 break

      end

      prev = current

   end

   -- these will be assigned to the inner and outer aperture info

   -- make sure we found a match
   if not match then
      local err = string.format( "%s: unable to find mirror '%s%d' in '%s'", modname, optic, shell, args.db )
	 logger:error( err )
	 error( err )

      -- no smaller mirror
   elseif not inner then

      -- if there's only one mirror then inner_disk must be specified
      if not outer and not args.inner_disk then
	 local err = string.format( "%s: could not find neighboring mirror to '%s%d' in '%s' and inner_disk was not specified", modname, optic, shell, args.db )
	 logger:error( err )
	 error( err  )
      end

      -- if inner_disk is specified use that
      if args.inner_disk then

	 logger:info( "%s: mirror '%s%d':  Using inner_disk", modname, optic, shell )

	 inner = args.inner_disk

	 -- not specified, use the distance to the next outermost mirror
      else

	 logger:info( "%s: mirror '%s%d': No smaller mirror found in '%s'; using distance to '%s%d'", modname, optic, shell,args.db, outer.optic, outer.shell )

	 inner = tables.copy( match )
	 inner.r = match.r - ( outer.r - (match.r + match.rthick ) )

      end

      -- next smaller was specified
   else

      logger:info( "%s: mirror '%s%d': Using distance to '%s%d'", modname, optic, shell, inner.optic, inner.shell )
      inner.r = inner.r + inner.rthick

   end

   logger:info( "%s: mirror '%s%d': entrance aperture radii = ( %f, %f )", modname, optic, shell, inner.r, match.r )

   return inner, match

end

return M
